#!/bin/sh

# ====================================================================
# Launch script for Deus Ex Mankind Divided Release
# Thu Oct  6 17:03:31 UTC 2016

# If you have useful edits made for unsupported distros then please
# let us know at <linuxscriptsuggestions@feralinteractive.com>

# Variables
FERAL_GAME_NAME="DeusExMD"
FERAL_GAME_NAME_FULL="Deus Ex Mankind Divided"
FERAL_GAME_STEAMID="337000"
FERAL_LIB_PATH="lib/x86_64"

# 'Magic' to get the game root
GAMEROOT="$(cd "${0%/*}" && echo "$PWD")"

# Check for arguments
# Note: some of these can be set at a system level to override for
# all Feral games
while [ $# -gt 0 ]; do
	arg=$1
	case ${arg} in
		--fresh-prefs)   FERAL_FRESH_PREFERENCES=1  && shift ;;
		--glc-capture)   FERAL_GLC_CAPTURE=1        && shift ;;
		--system-asound) FERAL_SYSTEM_ASOUND=1      && shift ;;
		--version)       FERAL_GET_VERSION=1        && shift ;;
		*) break ;;
	esac
done

# ====================================================================
# Options

# Automatically backup old preferences and start fresh on launch
if [ "${FERAL_FRESH_PREFERENCES}" = 1 ]; then
	mv "$HOME/.local/share/feral-interactive/${FERAL_GAME_NAME_FULL}" "$HOME/.local/share/feral-interactive/${FERAL_GAME_NAME_FULL}-$(date +%Y%m%d%H%M%S).bak"
fi

# Run with glc as the debugger for screen capture
if [ "${FERAL_GLC_CAPTURE}" = 1 ]; then
	GLCFILE="${FERAL_GAME_NAME}-$(date +%Y%m%d%H%M%S)"
	mkdir -p /tmp/ScreenCaptures && GAME_LAUNCH_PREFIX="glc-capture -o /tmp/ScreenCaptures/${GLCFILE}.glc"
fi

# Show a version panel on start
if [ "${FERAL_GET_VERSION}" = 1 ]; then
	HAS_ZENITY="$(command -v zenity)"
	if [ -n "${HAS_ZENITY}" ]; then
		zenity --text-info --title "${FERAL_GAME_NAME_FULL} - Version Information" --filename "${GAMEROOT}/share/FeralInfo.json"
	else
		echo "==================== ${FERAL_GAME_NAME_FULL} - Version Information ===================="
		cat "${GAMEROOT}/share/FeralInfo.json"
	fi
	exit
fi

# ====================================================================
# Our games are compiled targeting the steam runtime and are not
# expected to work perfectly when run outside of it
# However on some distributions (Arch Linux/openSUSE etc.) users have
# had better luck using their own libs
# Comment these lines out if testing that
if [ -z "${SteamAppId}" ]; then
	echo "WARNING: ${FERAL_GAME_NAME_FULL} not launched from steam"
	echo "         This is likely incorrect"
	echo "         Launching steam in 3 seconds with steam://rungameid/${FERAL_GAME_STEAMID}"
	sleep 3
	steam "steam://rungameid/${FERAL_GAME_STEAMID}"
	exit
fi

# ====================================================================
# Set the steam appid if not set
if [ "${SteamAppId}" != "${FERAL_GAME_STEAMID}" ]; then
	SteamAppId="${FERAL_GAME_STEAMID}"
	export SteamAppId
fi

# ====================================================================
# Enviroment Modifiers

# Store the current LD_PRELOAD
SYSTEM_LD_PRELOAD="${LD_PRELOAD}"
LD_PRELOAD_ADDITIONS=

# Unset LD_PRELOAD temporarily
# This avoids a chunk of confusing 32/64 errors from the steam overlay
# It also allows us to call the system openssl and curl here
# If your distribution needed an LD_PRELOAD addition then it should be
# fine to comment this out
unset LD_PRELOAD

# LC_ALL has caused users many issues in the past and generally is just
# used for debugging
# Uncomment this line if LC_ALL was needed (sometimes on openSUSE)
unset LC_ALL

# Try and set up SSL paths for all distros, due to steam runtime bug #52
# The value is used by our version of libcurl
# Users on unsupported distros might want to check if this is correct
HAS_CURL="$(command -v curl-config)"
if [ -n "${HAS_CURL}" ]; then
	SSL_CERT_FILE="$(curl-config --ca)"
	export SSL_CERT_FILE
else
	# Otherwise try with guess work
	if [ -e /etc/ssl/certs/ca-certificates.crt ]; then
		SSL_CERT_FILE="/etc/ssl/certs/ca-certificates.crt"
		export SSL_CERT_FILE
	elif [ -e /etc/pki/tls/certs/ca-bundle.crt ]; then
		SSL_CERT_FILE="/etc/pki/tls/certs/ca-bundle.crt"
		export SSL_CERT_FILE
	elif [ -e /var/lib/ca-certificates/ca-bundle.pem ]; then
		SSL_CERT_FILE="/var/lib/ca-certificates/ca-bundle.pem"
		export SSL_CERT_FILE
	fi
fi
HAS_OPENSSL="$(command -v openssl)"
if [ -n "${HAS_OPENSSL}" ]; then
	SSL_CERT_DIR="$(openssl version -d | sed -E 's/.*\"(.*)\"/\1/')/certs"
	export SSL_CERT_DIR
fi

# AMD fglrx drivers run faster with TCMalloc
HAS_CATALYST="$(grep fglrx /proc/modules)"
if [ -n "${HAS_CATALYST}" ]; then
	LD_PRELOAD_ADDITIONS="../${FERAL_LIB_PATH}/libtcmalloc_minimal.so:${LD_PRELOAD_ADDITIONS}"
fi

# Brute force fix for some small thread sizes in external libraries
if [ -e "${GAMEROOT}/${FERAL_LIB_PATH}/libminimum_thread_stack_size_wrapper.so" ]; then
	LD_PRELOAD_ADDITIONS="../${FERAL_LIB_PATH}/libminimum_thread_stack_size_wrapper.so:${LD_PRELOAD_ADDITIONS}"
fi

# Use the system asound if requested
# This can help with sound issues on some distros including Arch Linux
if [ "${FERAL_SYSTEM_ASOUND}" = 1 ]; then
	LIBASOUND_DYLIB="libasound.so.2"
	if [ -e "/usr/lib/x86_64-linux-gnu/${LIBASOUND_DYLIB}" ]; then
		LIBASOUND_LIBDIR="/usr/lib/x86_64-linux-gnu"
	elif [ -e "/usr/lib64/${LIBASOUND_DYLIB}" ]; then
		LIBASOUND_LIBDIR="/usr/lib64"
	elif [ -e "/usr/lib/${LIBASOUND_DYLIB}" ]; then
		LIBASOUND_LIBDIR="/usr/lib"
	fi
	LD_PRELOAD_ADDITIONS="${LIBASOUND_LIBDIR}/${LIBASOUND_DYLIB}:${LD_PRELOAD_ADDITIONS}"
fi

# Add our additionals and the old preload back
LD_PRELOAD="${LD_PRELOAD_ADDITIONS}:${SYSTEM_LD_PRELOAD}"
export LD_PRELOAD

# ====================================================================
# Run the game
cd "${GAMEROOT}/bin" && ${GAME_LAUNCH_PREFIX} "${GAMEROOT}/bin/${FERAL_GAME_NAME}" "$@"
RESULT=$?

# ====================================================================
# Options

# Automatically convert the glc file to mp4, must be pared with command above
if [ -e "/tmp/ScreenCaptures/${GLCFILE}.glc" ]; then
	unset LD_PRELOAD
	unset LD_LIBRARY_PATH
	glc-play "/tmp/ScreenCaptures/${GLCFILE}.glc" -o - -y 1 | ffmpeg -i - -y "$HOME/.local/share/feral-interactive/${FERAL_GAME_NAME_FULL}/${GLCFILE}.mp4"
fi

# ====================================================================
exit "${RESULT}"
