#!/bin/sh

# Force C locale
LC_ALL=C
export LC_ALL

SAVED_LD_LIBRARY_PATH=$LD_LIBRARY_PATH
unset LD_LIBRARY_PATH
INSTALLDIR=$1
OUTFILE="$( cd "$( dirname "$2" )"; pwd )/$( basename "$2" )"
FERAL_PREFS="$XDG_DATA_HOME/feral-interactive"
if [ ! -d $FERAL_PREFS ]; then
	FERAL_PREFS="$HOME/.local/share/feral-interactive"
fi

# If we're running from the PGOW, don't close tags so more info can be added.
PGOWAPPEND=$3

# --------------------------------------------------------------------------------
output_text() {
	echo $1
	echo "<h3 id=\"$1\">$1</h3>" >> "$OUTFILE"
} 

# --------------------------------------------------------------------------------
if [ ! -n "$OUTFILE" ] || [ ! -n "$INSTALLDIR" ]; then
	echo "USAGE: \$ $0 game_install_dir output_file_path"
	exit 1
fi
if [ ! -d "$INSTALLDIR" ]; then
	echo "$INSTALLDIR not found, game install missing?"
	exit 1
fi

# --------------------------------------------------------------------------------
TEXT_AREA="<textarea rows=\"5\" readonly>"

# --------------------------------------------------------------------------------
echo "Reporting with $INSTALLDIR to $OUTFILE"
echo "<!DOCTYPE html>
<html>
<head>
<title>Feral System Report</title>
<style>
textarea {
    width:90%;
    height:100px;
}
</style>
</head>
<body>
<h1>Feral System Report</h1>
<p>Generated using '\$ $0 $*' at $(date)</p>
<hr>
<h3>Contents</h3>
<p>
<a href=\"#programs\">Program Outputs</a><br>
<a href=\"#system\">System Files</a><br>
<a href=\"#installed\">Installed Files</a><br>
<a href=\"#preferences\">Preferences</a><br>" >> "$OUTFILE"

# Add steam dlc info from C++.
if [ "$PGOWAPPEND" = "PGOWAppend" ]; then
	echo "<a href=\"#steamdlc\">Steam DLC Info</a><br>" >> "$OUTFILE"
fi

echo "</p>" >> "$OUTFILE"


WRITE_ERROR=$?
if [ $WRITE_ERROR != 0 ] ; then
	exit $WRITE_ERROR
fi

# Title
echo "<hr><h2 id=\"programs\">Program Outputs</h2>" >> "$OUTFILE"

# --------------------------------------------------------------------------------
# "uname -a"           - System and kernel version info
# "lsb_release -a"     - More specific system info
# "nvidia-smi"         - Current GPU stats on nvidia
# "fglrxinfo"          - Current GPU stats on amd
# "lspci -v"           - Info on current hardware
# "lsusb -v"           - Info on USB devices
# "env"                - Check against steam runtime enviroment to catch oddities
# "top -b -n 1"        - Running processes (useful to detect CPU/GPU hogs or zombie processes)
# "glxinfo"            - Detailed opengl information
# "setxkbmap -query"   - Information on current keyboard map/modes
# "curl-config --ca"   - Location of the certificates bundle
set --  "uname -a" \
	"lsb_release -a" \
	"nvidia-smi" \
	"fglrxinfo" \
	"lspci -v" \
	"lsusb -v" \
	"env" \
	"top -b -n 1" \
	"glxinfo" \
	"setxkbmap -query" \
	"curl-config --ca"
for CMD do
	output_text "$CMD"
	echo "${TEXT_AREA}" >> "$OUTFILE"
	if [ "$CMD" = "env" ]; then
		LD_LIBRARY_PATH=$SAVED_LD_LIBRARY_PATH
		export LD_LIBRARY_PATH
	fi
	$CMD 2>&1 | head -n 500 | tee -a "$OUTFILE" | 
	if [ "$(wc -l)" = "500" ]; then 
		echo "...truncated..." >> "$OUTFILE" 
	fi
	if [ "$CMD" = "env" ]; then
		unset LD_LIBRARY_PATH
	fi
	echo "</textarea>" >> "$OUTFILE"
done


# Title
echo "<hr><h2 id=\"system\">System Files</h2>" >> "$OUTFILE"

# --------------------------------------------------------------------------------
# "/etc/*-release"                   - Info on system release version
# "/etc/X11/default-display-manager" - X11 display manager info
# "/proc/meminfo"                    - Info on current RAM
# "/proc/cpuinfo"                    - Info on current CPU
# "/etc/sysconfig/displaymanager"    - Display manger config
# "/etc/sysconfig/desktop"           - WM config
# "/proc/bus/input/devices"          - input devices (controllers + m/k)
set --  $(echo "/etc/*-release") \
	"/etc/X11/default-display-manager" \
	"/proc/meminfo" \
	"/proc/cpuinfo" \
	"/etc/sysconfig/displaymanager" \
	"/etc/sysconfig/desktop" \
	"/proc/bus/input/devices" 
for FILE do
	if [ -e "$FILE" ] ; then
		output_text "$FILE"
		echo "${TEXT_AREA}" >> "$OUTFILE"
		cat "$FILE" 2>&1 | head -n 500 | tee -a "$OUTFILE" | 
		if [ "$(wc -l)" = "500" ]; 
			then echo "...truncated..." >> "$OUTFILE"; 
		fi
		echo "</textarea>" >> "$OUTFILE"
	else
		output_text "$FILE not found"
	fi
done


# Title
echo "<hr><h2 id=\"installed\">Installed Files</h2>" >> "$OUTFILE"

# --------------------------------------------------------------------------------
# "ls -lRh"            - full information on current installed game files
cd "$INSTALLDIR"
output_text "ls -lRh in '$INSTALLDIR'"
echo "${TEXT_AREA}" >> "$OUTFILE"
ls -lRh >> "$OUTFILE" 2>&1
echo "</textarea>" >> "$OUTFILE"

# --------------------------------------------------------------------------------
# "$INSTALLDIR/*.sh"                 - Launch script(s)
# "$INSTALLDIR/share/*.json"         - Configuration JSON files
# "$INSTALLDIR/share/*.xml"          - Configuration XML files
# "$INSTALLDIR/share/*.txt"          - Configuration TXT files
cd "$INSTALLDIR"
for FILE in *.sh share/*.json share/*.xml share/*.txt
do
	output_text "'$FILE' in '$INSTALLDIR'"
	echo "${TEXT_AREA}" >> "$OUTFILE"
	cat "$FILE" 2>&1 | head -n 500 | tee -a "$OUTFILE" | 
	if [ "$(wc -l)" = "500" ]; 
		then echo "...truncated..." >> "$OUTFILE"
	fi
	echo "</textarea>" >> "$OUTFILE"
done

# Title
echo "<hr><h2 id=\"preferences\">Preferences</h2>" >> "$OUTFILE"

# --------------------------------------------------------------------------------
# "$FERAL_PREFS/*/preferences"       - Preferences files
cd "$FERAL_PREFS"
for FILE in */preferences
do
	output_text "'$FILE' in '$FERAL_PREFS'"
	echo "${TEXT_AREA}" >> "$OUTFILE"
	cat "$FILE" 2>&1 | head -n 500 | tee -a "$OUTFILE" | 
	if [ "$(wc -l)" = "500" ]; 
		then echo "...truncated..." >> "$OUTFILE"
	fi
	echo "</textarea>" >> "$OUTFILE"
done

if [ "$PGOWAPPEND" != "PGOWAppend" ]; then
	echo "</body>
	</html>" >> "$OUTFILE"
fi
