local curAnimDataBasePath
local curAnimSetPath
local curAnimSet

-- A character type is defined as a name, potentially followed by a table of child character types.
local numCharacterTypes = 0
local charTypes = {name = "Root", depth = -1, children = {}, parent = nil}
local curCharType = nil
local variationAnimList = {}

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
-- PRIVATE FUNCTIONS
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 

local function CountCharacters(str, character)
	local searchPattern = string.format("%%%s", character)
	local count = 0
	for c in string.gfind(str, searchPattern) do
		count = count + 1
	end
	
	return count
end


local function ParseSingleCharType(filePointer, currentType)
	local curLine = filePointer:read("*l")
	if(curLine == nil) then
		return
	end

	-- Find the name and depth of the next character type
	local curDepth = CountCharacters(curLine, "+")

	-- Create a new character type
	local charTypeName = string.sub(curLine, curDepth+1)
	if(charTypeName == nil or string.len(charTypeName) <= 0) then
		return
	end
	
	local newType = { name = charTypeName, depth = curDepth, children = {}, parent = nil}
	numCharacterTypes = numCharacterTypes + 1

	-- Determine where in the hierarchy this character type fits
	if(curDepth > currentType.depth) then
		-- The new character type is a child of the current type
		--print("Adding "..charTypeName.." as a child of "..currentType.name)
		newType.parent = currentType
		table.insert(currentType.children, newType)
	else
		-- The new character type is on the same level or higher up than the current type    
		local curEntry = currentType
		local placeFound = false
		while(not placeFound) do
			if(curEntry.depth == newType.depth) then
				newType.parent = curEntry.parent
				--print("Adding "..newType.name.." as a child of "..curEntry.parent.name)
				table.insert(curEntry.parent.children, newType)
				placeFound = true
			else
				-- Take one step up the hierarchy to reach desired depth
				curEntry = curEntry.parent
			end
		end
	end
	ParseSingleCharType(filePointer, newType)
end

local function FindCharType(charType, charTypeName)
	if(charType.name == charTypeName) then
		-- Macth found.
		return charType
	else
		-- Not a match. Look through all children.
		for i = 1, table.getn(charType.children), 1 do
			local result = FindCharType(charType.children[i], charTypeName)
			if(result ~= nil) then
				return result
			end
		end
	end
end

local function ConvertAnimSetVariationToCharacterTypes(variation, parentType)
	local newType = { name = variation.Name, depth = parentType.depth + 1, children = {}, parent = parentType }
	numCharacterTypes = numCharacterTypes + 1

	table.insert(parentType.children, newType)

	for i,child in variation.Children do
		ConvertAnimSetVariationToCharacterTypes(child, newType)
	end
end

local function SaveVariationResourceIDInfo()
	-- This saves a variation resource ID based on the current anim data base and selected character type
	-- The file will be stored in project:\temp\MC_CharacterType\VariationResourceID.txt for the RTT to find.
	-- NB: We might to save this info on a per network basis. Otherwise the default value will be set everytime you open it.

	local variationFilePath = projectPath.."\\temp\\MC_CharacterType\\VariationResourceID.txt"

	local variationResourceID = string.gsub(utils.demacroizeString(curAnimSetPath), nocase(assemblyPath), "assembly:/")
	variationResourceID = string.format("%s?/%s.asva", variationResourceID, curCharType.name)

	print("WRITING variation resource ID: " .. variationResourceID)

	local fh, error = io.open(variationFilePath, "w+")
	if(fh ~= nil) then
		-- Write the resource identifier
		fh:seek("set")
		fh:write(variationResourceID)
		fh:close()
	else
		ReportError(string.format("Could not store variation resourceID to %s.", variationFilePath))
	end
end

-- Function for listing parsed characetr types. Mainly for debugging purposes.
local function PrintCharacterTypes(charType)
	for k1, v1 in charType do
		if(type(v1) ~= "table") then
			print(k1..": "..v1)
		elseif(k1 == "children") then
			print("Listing children:\n")
			for k2, v2 in v1 do
				PrintCharacterTypes(v2)
			end
		elseif(k1 == "parent") then
			print(k1..": "..v1.name)
		end
	end
end

-- Function for listing animation variations for the current character type
local function PrintAnimVariations()
	local numVariations = 0
	for k1, v1 in variationAnimList do
		numVariations = numVariations + 1
		print(k1..": "..v1.AnimPath..", "..v1.TakeName)
	end
	
	print(string.format("Found %i variations for character type %s", numVariations, curCharType.name))
end

local function FindVariationInAnimSet(variation, name)
	local variationName = string.lower(variation.Name)
	if variationName == name then
		return variation
	else
		for i,child in variation.Children do
			local found = FindVariationInAnimSet(child, name)	
			if found ~= nil then
				return found
			end
		end
		return nil
	end
end

local function FindVariationAnimation(variation, definition)
	for i, entry in variation.Entries do
		if entry.Definition == definition and entry.Animations ~= nil and table.getn(entry.Animations) > 0 then
			return entry.Animations[1]
		end
	end

	if variation.Parent ~= nil then
		return FindVariationAnimation(variation.Parent, definition)
	end

	return nil
end

local function BuildVariationAnimList()
	if curAnimSet == nil then
		ReportError("BuildVariationAnimList(): curAnimSet is nil!")
		return false
	end

	-- Clear the previous variation table
	variationAnimList = {}

	-- Find the variation
	local variation = FindVariationInAnimSet(curAnimSet.DefaultVariation, string.lower(curCharType.name))
	if variation == nil then
		ReportError("BuildVariationAnimList(): could not find variation:" .. curCharType.name)
		return false
	end

	for i,definition in curAnimSet.EntryDefinitions do
		local animation = FindVariationAnimation(variation, definition.Name)
		if animation ~= nil then
			local path, takeName = string.match(animation, "assembly:([^?]*)?/(.*)")
			if path ~= nil then
				path = "$(RootDir)\\Assembly" .. string.gsub(path, "/", "\\")
				takeName = string.gsub(takeName, nocase(".xmdtake"), "")
				variationAnimList[definition.Name] = { AnimPath = path, TakeName = takeName }
			end
		end
	end

	return true
end


-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- PUBLIC FUNCTIONS
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

function GetCurrentAnimDBPath()
	return curAnimDataBasePath
end

function GetCurrentCharType()
	return curCharType
end

function SetAnimDataBase(dataBasePath)
	curAnimDataBasePath = nil
	curAnimSetPath = nil
	curAnimSet = nil

	-- Reset character type data
	numCharacterTypes = 0
	charTypes = {name = "Root", depth = -1, children = {}, parent = nil}
	curCharType = nil
	variationAnimList = {}
	curAnimDataBasePath = dataBasePath
	return SetAnimSet(dataBasePath)
end

local function SetVariationParentReferences(variation)
  for i, child in variation.Children do
	child.Parent = variation;
	SetVariationParentReferences(child)
  end
end

function SetAnimSet(animSetPath)
	curAnimSetPath = animSetPath

	local fh, error = io.open(utils.demacroizeString(animSetPath), "rb")
	if(fh ~= nil) then
		local content = fh:read("*a")
		curAnimSet = JSON:decode(content)

		-- Fix up the data so we also have reference from children to parent
		SetVariationParentReferences(curAnimSet.DefaultVariation)
	else
	  ReportError("SetAnimDataBase(): Could not load animaton set: " .. curAnimSetPath)
	  return false
	end
	
	ConvertAnimSetVariationToCharacterTypes(curAnimSet.DefaultVariation, charTypes)
	return true
end


function SetCurrentCharType(charTypeName)
	local result = FindCharType(charTypes, charTypeName)
	if(result ~= nil) then
		curCharType = result
		SaveVariationResourceIDInfo()
		return BuildVariationAnimList()
	else
		return false
	end
end

function GetCharTypes()
	-- Ignore the root here
	return charTypes.children
end


function SetDefaultCharacterType()
	if(not SetCurrentCharType(defaultCharacterType)) then
		local fallbackCharacterType = charTypes.children[1].name	
		if(fallbackCharacterType == nil or string.len(fallbackCharacterType) <= 0) then
			ReportError(string.format("Fallback character type \"%s\" is invalid. Unable to set any character type.", tostring(fallbackCharacterType)))
			return
		end

		local fallbackWasSet = SetCurrentCharType(fallbackCharacterType)
		if(not fallbackWasSet) then
			ReportError(string.format("Failed to set fallback character type \"%s\". Unable to set any character type.", fallbackCharacterType))
			return
		end

		ReportWarning(string.format("Default character type \"%s\" was not found in the Animation Database! Using \"%s\" instead.", defaultCharacterType, fallbackCharacterType))
	end
end

function GetVariationEntry(variationName)
	return variationAnimList[variationName]
end


function CreateTempCharacterTypeDir()
	local charTypeFileDir = projectPath.."\\temp\\MC_CharacterType"
	local mkdirCommandLine = string.format("mkdir %s", charTypeFileDir)
	app.execute(mkdirCommandLine, false, true)
end

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
-- INITIALIZATION
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

CreateTempCharacterTypeDir()
SetAnimDataBase(defaultAnimDataBasePath)
SetDefaultCharacterType()