-- This file is intended to function as a sort of interface to the custom data
-- files stored along with each network(.mcn file). Other scripts can request
-- specific custom information related to their field via global functions
-- defined in this script. All handling of the custom data files are handled
-- by this script.


customData = {}

local READ					= 1
local WRITE					= 2
local APPEND				= 3
local UPDATE_PRESERVE		= 4
local UPDATE_ERASE			= 5
local UPDATE_APPEND_ONLY	= 6

local fileModes =
{
	[READ] = "r", 
	[WRITE] = "w", 
	[APPEND] = "a",
	[UPDATE_PRESERVE] = "r+",
	[UPDATE_ERASE] = "w+",
	[UPDATE_APPEND_ONLY] = "a+"
}

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- PRIVATE FUNCTIONS
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
local function OpenFile(filePath, mode)
	-- This function attempts to open a file on disk. If the file does not exist, one is created automatically.
	local fh, error = io.open(filePath, fileModes[mode])
	if(fh == nil) then
		if(string.find(error, "No such file or directory") ~= nil) then
			-- The file does not exist on disk, so we must create one	
			fh, error = io.open(filePath, fileModes[UPDATE_ERASE])
			if(fh == nil) then
				ReportError(string.format("Failed to create file: %s\n\nError: %s\n", filePath, error))
			else		
				-- We have created a new file on the disk. Close the current file handle, and reopen it in the desired mode
				fh:close()
				fh, error = io.open(filePath, fileModes[mode])
				if(fh == nil) then
					-- There was some error, other than a non-existent file.
					ReportError("Failed to open custom data file: \n"..error)
				end
			end
		else
			-- There was some error, other than a non-existent file.
			ReportError("Failed to open custom data file: \n"..error)
		end
	end
	
	return fh, error
end 

local function GetCustomDataFileHandle(layerFilePath, mode)
	
	local mcnFilePath = mcn.filename()
	if(mcnFilePath == nil or mcnFilePath == "") then
		return nil
	end
	
	-- Make sure that we are asking for a file in either the construction or assembly layer
	local useConstructionLayer = (layerFilePath == constructionPath)
	local useAssemblyLayer = (layerFilePath == assemblyPath)
	if(not useConstructionLayer and not useAssemblyLayer) then
		ReportError(string.format("Custom data can only be loaded from either the construction or assembly layer. Requested path was:", layerFilePath))
		return nil
	end
	
	-- Build a folder name based on the network file name
	local temp = string.gsub(mcnFilePath,  nocase(constructionPath.."AnimationNetworks\\"), "")
	local customDataFolderName = "NetworkRoot"
	local customDataFileName = ""
	for subFolder in string.gfind(temp, "[^\\]+") do
		if(not string.find(subFolder, "%.")) then
			customDataFolderName = customDataFolderName.."_"..subFolder
		else
			customDataFileName = string.gsub(subFolder, nocase(".mcn"), ".cstmdata")
		end
	end

	local customDataFilePath = layerFilePath.."\\AnimationNetworks\\_CUSTOM_DATA\\"..customDataFolderName
	
	-- If the directory does not already exists, we need to create it.
	app.execute("mkdir "..quoteString(customDataFilePath), false, true)
	
	-- Finally append the file name itself
	customDataFilePath = customDataFilePath.."\\"..customDataFileName
	
	-- If we are opening the file for writing, we need to do some source control juggling here
	local openFile = true
	local addCustomDataFileToSourceControl = false
	if(mode ~= READ) then
		if(useAssemblyLayer) then
			-- Files in the assembly layer are under source control
			if(IsFileInDepot(customDataFilePath)) then
				if(IsFileInDepot(mcnFilePath)) then
					if(IsFileInOurChangeList(mcnFilePath)) then
						openFile = CheckOutFile(customDataFilePath)
						if(openFile) then
							LockFile(customDataFilePath)
						end
					else
						app.log("Cannot checkout custom data file in the assembly layer. Check out the network (.mcn) file to do so.")
						openFile = false
					end
				else
					ReportError(".cstmdat files should not be under source control for networks not in the depot!")
					openFile = false
				end
			else
				if(IsFileInDepot(mcnFilePath)) then
					if(IsFileInOurChangeList(mcnFilePath)) then
						-- Even though it is not under source control a cstmdata file might still exist on disk. If so we add that otherwise we have to create a new file first
						local fh = OpenFile(customDataFilePath, READ)
						if(fh ~= nil) then
							fh:close()
							addCustomDataFileToSourceControl = true
						end
					else
						app.log("Cannot create a custom data file. Check out the network (.mcn) file to do so.")
						openFile = false
					end
				end
			end
		end
	end
	
	if(openFile) then
		-- Add the custom data file to source control if requested
		if(addCustomDataFileToSourceControl) then
			OpenForAdd(customDataFilePath, true)
		end
		
		-- Open the custom data file and return the file handle
		local fh, error = OpenFile(customDataFilePath, mode)
		if(fh == nil) then
			ReportError("Failed to create custom data file: \n"..error)
			return nil
		else
			fh:seek("set")
			return fh
		end
	else
		-- The custom data file can not be accessed due to some earlier problem, so return nil
		return nil
	end
end

local function LoadCustomData(fileHandle)
	-- This reads all custom data from a given file handle and stores it in the table customData.
	if(fileHandle == nil) then
		ReportError("Could not load custom data. Invalid file handle.")
		return false
	end
	
	-- Erase any previously stored custom data. Is this the correct way to delete a table? Just set it to empty and let the garbage collector worry about it?
	customData = {}
	fileHandle:seek("set")
	for line in fileHandle:lines() do
		-- Read the entry key as all upper case letters
		local startIndex, endIndex, entryKey = string.find(line, "([%u_]+)")
		if(entryKey == nil) then
			ReportError(string.format("Custom data file line: %s Did not contain a properly formed upper case entry key."))
			return false
		end
		local value = string.sub(line, endIndex+2)
		customData[entryKey] = value
	end
	
	return true
end

local function SaveCustomData(fileHandle)
	-- This writes whatever is stored in the custom data table to a file pointed to by fileHandle
	if(fileHandle == nil) then
		ReportError("Could not save custom data. Invalid file handle.")
		return false
	end
	
	fileHandle:seek("set")
	for k, v in pairs(customData) do
		fileHandle:write(string.format("%s %s\n", k, v))
	end
	fileHandle:flush()
	return true
end

local function SaveCustomDataEntry(layerPath, entryKey, entryValue)
	local fh = GetCustomDataFileHandle(layerPath, READ)
	if(fh == nil) then
		return false
	end
	
	local loadOK = LoadCustomData(fh)
	fh:close()
	if(not loadOK) then
		return false
	end
	
	-- Write custom data to disk
	fh = GetCustomDataFileHandle(layerPath, UPDATE_ERASE)
	if(fh == nil) then
		return false
	end
	
	-- Set the custom data entry
	customData[entryKey] = entryValue	
	local saveOK = SaveCustomData(fh)
	fh:close()
	
	return saveOK
end

local function LoadRig(fileHandle)
	-- Load up the value stored under RIG_PATH in the file pointed to by fileHandle
	if(not LoadCustomData(fileHandle)) then
		return nil
	end
	
	if(customData.RIG_PATH == nil) then
		return nil
	else
		return utils.demacroizeString(customData.RIG_PATH)
	end
end


local function GetRigPath(layerPath)
	local fh = GetCustomDataFileHandle(layerPath, READ)
	if(fh == nil) then
		return nil
	end
	
	local rigPath = LoadRig(fh)
	fh:close()
	
	return rigPath
end


-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- PUBLIC FUNCTIONS
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
function LoadCharacterType()
	-- Load the stored name of the character type associated with the mcn file
	local fh = GetCustomDataFileHandle(constructionPath, READ)
	if(fh == nil) then
		return false
	end
	
	local loadOK = LoadCustomData(fh)
	fh:close()
	if(not loadOK) then
		return false
	end

	if(customData.CHARACTER_TYPE == nil) then
		return SetDefaultCharacterType()
	end

	local charTypeWasSet = SetCurrentCharType(customData.CHARACTER_TYPE)
	if(not charTypeWasSet) then
		ReportWarning(string.format("Character type \"%s\" was not found in the Animation Database. Using default character type instead.", customData.CHARACTER_TYPE))
		return SetDefaultCharacterType()
	end
end


function SaveCharacterType()	
	-- When setting a character type, we also save the animation database that it belongs to to ensure consistency
	if(not SaveAnimDataBasePath()) then
		return false
	end
	
	local curCharType = GetCurrentCharType()
	return SaveCustomDataEntry(constructionPath, "CHARACTER_TYPE", curCharType.name)
end

function LoadAnimDataBasePath()
	local fh = GetCustomDataFileHandle(constructionPath, READ)
	if(fh == nil) then
		return false
	end
	
	local loadOK = LoadCustomData(fh)
	fh:close()
	if(not loadOK) then
		return false
	end
	
	if(customData.ANIM_DATABASE_PATH == nil) then
		app.log(string.format("No animation database set for this network. Setting default animation database."))
		return SetAnimDataBase(defaultAnimDataBasePath)
	end
	
	local animDBWasSet = SetAnimDataBase(customData.ANIM_DATABASE_PATH)
	if(not animDBWasSet) then
		ReportWarning(string.format("Animation database \"%s\" could not be set. Using default animation database.", customData.ANIM_DATABASE_PATH))
		return SetAnimDataBase(defaultAnimDataBasePath)
	end	
end

function SaveAnimDataBasePath()
	local curAnimDBPath = GetCurrentAnimDBPath()
	return SaveCustomDataEntry(constructionPath, "ANIM_DATABASE_PATH", curAnimDBPath)
end

function LoadPreviewRig()
	local rigPath = GetRigPath(constructionPath)
	if(rigPath == nil) then
		return false
	end
	
	return SetPreviewRig(rigPath)
end

function SavePreviewRig()
	local previewRig, exportRig = GetRigPaths()
	return SaveCustomDataEntry(constructionPath, "RIG_PATH", utils.macroizeString(previewRig))
end

function LoadExportRig()
	local rigPath = GetRigPath(assemblyPath)
	if(rigPath == nil) then
		return false
	end
	
	return SetExportRig(rigPath)
end

function SaveExportRig()
	local previewRig, exportRig = GetRigPaths()
	return SaveCustomDataEntry(assemblyPath, "RIG_PATH", utils.macroizeString(exportRig))
end


-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- INITIALIZATION
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --