local function GetExportDir(fileName)
	local exportDir = string.gsub(fileName, nocase(constructionPath), assemblyPath)
	return string.sub(exportDir, 0, string.find(exportDir, "[\\/][^\\/]*$"))	
end

local function ResolveExportLocation(fileName, sourceExt, targetExt)
	
	-- Get the network name
	local networkName = fileName
	local index = string.find(networkName, "\\")
	while (index ~= nil) do
		networkName = string.sub(networkName, index+1)
		index = string.find(networkName, "\\")
	end

	networkName = string.sub(networkName, 0, string.find(networkName, sourceExt)-2) -- For some reason searching for the "." returns 1?

	-- Set up a path to the export directory
	local exportDir = GetExportDir(fileName)
	local targetFilePath = string.format("%s%s.%s", exportDir, networkName, targetExt)
	
	return networkName, exportDir, targetFilePath
end

local function ExportNetwork(outputFileName)

	-- Before setting the export rig, we need to deselect any animation in the animation bowser.
	-- Otherwise the animation will be packed against the export rig, which will fail since the export
	-- does not have a wl2 primitive file associated with it.
	local animInfo = anim.getSelectedTakeInBrowser()	
	local selectedAnimPath = animInfo.filename
	local selectedTakeName = animInfo.takename
	
	-- Deselect anything in the anim browser
	anim.selectTakeInBrowser("")
	
	-- Switch out the preview rig with export rig.
	local previewRig, exportRig = GetRigPaths()
	if(not anim.setRigPath(utils.macroizeString(exportRig))) then
		local errorTable = {[1] = {["name"] = "RigError"; ["message"] = "Failed to set export rig."}}
		return false, errorTable
	end
	mcn.commit("Export rig set")
	
	local result, ids, errors, warnings = mcn.export(outputFileName)
	showNetworkValidationReport(ids, warnings, errors)
	
	--[[
	-- Print runtime ids
	print("Runtime IDs:")
	for i, v in ipairs(ids) do
		print(string.format("%s -> %s", v.name, v.id))
	end
	--]]
	
	-- Reset the rig back to the preview one
	if(not anim.setRigPath(utils.macroizeString(previewRig))) then
		result = false
		errors = {[1] = {["name"] = "RigError"; ["message"] = "Failed to reset the preview rig."}}
	end
	mcn.commit("Preview rig reset")
	
	-- Re-select the animation in the browser if any
	if(selectedAnimPath ~= "") then
		local animationTake = { filename = selectedAnimPath, takename = selectedTakeName }
		anim.selectTakeInBrowser(animationTake)
	end
	
	return result, errors
end

local function GetFileLength(fileName)

	-- Read file length
	local fp, errorMessage = io.open(fileName, "rb") -- open in read mode
	assert(fp, errorMessage)
	local fileLength = fp:seek("end")
	fp:seek("set")
	fp:close()
	
	return fileLength
end

local function AppendFileContents(fileName, outFileHandle)

	-- Write the size of the appended file
	local fileLength = GetFileLength(fileName)
	outFileHandle:write(uint32ToASCII(fileLength))

	-- Write the content of the file (Renable when new LUA lib arrives)
	local fp, errorMessage = io.open(fileName, "rb")
	assert(fp, errorMessage)
	outFileHandle:write(fp:read(fileLength))
	fp:close()

	-- Remove xml file generated by connect
	local retVal, errorMessage
	local attemptCount = 0
	repeat
		retVal, errorMessage = os.remove(fileName)
		attemptCount = attemptCount +1
	until(retVal ~= nil or attemptCount >= 1000)
end

-- Combines network, animset, events and rig file into one xml file
local function CombineFiles(exportDir, exportName, combinedFilePath)

	print(string.format("CombineFiles(%s, %s, %s)", tostring(exportDir), tostring(exportName), tostring(combinedFilePath)))
	
	local pFile, errorMsg = io.open(combinedFilePath, "wb")
	assert(pFile, errorMsg)

	if (pFile ~= nil) then
		-- Write the aln file version no.
		pFile:write(string.format("%c", ALN_VERSION_NO))
		
		-- Write the network definition
		local currentFileName = string.format("%s%s.xml", exportDir, exportName)
		AppendFileContents(currentFileName, pFile)

		-- Write the animation library
		currentFileName = string.format("%s%s_Library.xml", exportDir, exportName)
		AppendFileContents(currentFileName, pFile)

		pFile:close()
	else
		print(string.format("Failed to open file %s: %s", combinedFileName, errorMsg))
	end
end


-- People are generally responsible for the source file themselves. Source control on the export level
-- only concerns itself with the exported file.
local ExportButtonFunc = function()	

	app.log("Exporting..")
	local currentFileName = mcn.filename()
	if(currentFileName == "") then
		ShowDialogue("Can't export", "Can't export. No network (.mcn) file is loaded.\nSave the network to disk if you haven't done that yet.\n", BUTTON_OK_ONLY)
		return
	end


	-- Set up a path to the export directory
	local networkName, exportDir, alnFilePath = ResolveExportLocation(currentFileName, "mcn", "aln")

	-- Get source control status for the target file
	local addToSrcCtrl = false
	local result, errMsg = IsFileInDepot(alnFilePath)
	if(errMsg ~= nil) then
		app.log("Failed to export to '"..alnFilePath.."': "..errMsg)
		ReportError(errMsg)
		return
	end
	
	if (not result) then
		addToSrcCtrl = true
	else
		-- Check out the file from source control
		local isCheckedOut = CheckOutFile(alnFilePath)
		if(not isCheckedOut) then
			local errMsg = string.format("Checkout of '%s' failed.", alnFilePath)
			app.log("Failed to export to '"..alnFilePath.."': "..errMsg)
			ReportError(errMsg)
			return
		end
	end		

	addMandatoryControlParameters()
	
	-- If the export directory does not already exists, we need to create it.
	app.execute("mkdir "..quoteString(exportDir), false, true)
	print("mkdir "..quoteString(exportDir))

	local exportPath = string.format("%s%s.xml", exportDir, networkName)
	local success, errors = ExportNetwork(exportPath)
	if(not success) then
		-- Open error dialog
		local errorMsg = string.format("Could not export %s:\n", currentFileName)
		for i, v in ipairs(errors) do
			errorMsg = string.format("%s\n%s: %s", errorMsg, v.name, v.message)
		end
		errorMsg = errorMsg.."\n"
		
		ShowDialogue("Failed to Export Network", errorMsg, BUTTON_OK_ONLY)
		return
	end
	
	os.remove(string.format("%snetworkHeader.h", exportDir)) -- Remove unused file created by Connect
	CombineFiles(exportDir, networkName, alnFilePath)
	
	if (addToSrcCtrl) then
		OpenForAdd(alnFilePath, true)
	end
	
	app.log("Export to '"..alnFilePath.."' is done.")
end

-- Add a custom export button to the table of UI button to be created
local function AddUIExportButton()
	
	local exportButton = {}
	exportButton.name = "G2Export"
	exportButton.imagePath = buildPath.."\\morpheme\\resources\\images\\IOI_Export.xpm"
	exportButton.funcPointer = ExportButtonFunc
	exportButton.helpText = "Export Network to an aln file"
	exportButton.enabled = true
	
	AddUIButton(exportButton)
end	

ALN_VERSION_NO = 4 -- If this is changed, it must be changed in alnhandler.h as well
AddUIExportButton()
