------------------------------------------------------------------------------------------------------------------------
-- Blend2MatchEvents node definition.
------------------------------------------------------------------------------------------------------------------------

registerNode("Blend2MatchEvents",
  {
    helptext = "Blends two animation streams together and time warps to match the syncronisation events",
    group = "Blends",
    image = "Blend2MatchEvents.png",
    id = generateNamespacedId(idNamespaces.NaturalMotion, 101),

    --------------------------------------------------------------------------------------------------------------------
    functionPins = 
    {
      ["Source0"] = {
                       input = true, 
                       array = false,
                       interfaces = {"Transforms", "Time", "Events"},
                     },

      ["Source1"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },

      ["Result"] = { 
                     input = false, 
                     array = false, 
                     interfaces = {"Transforms", "Time", "Events"}, 
                   },
    },

    dataPins = 
    {
       ["Weight"] = {
                      input = true, 
                      array = false, 
                      type = "float", 
                    },
       ["EventBlendingWeight"] = {
                      input = true, 
                      array = false, 
                      type = "float", 
                    },
    },

    pinOrder = {"Source0", "Source1", "Result", "Weight", "EventBlendingWeight"},

    --------------------------------------------------------------------------------------------------------------------
    attributes =
    {
      { name = "Loop", type = "bool", value = true },
      { name = "UseSource1Additively", type = "bool", value = true},
      { name = "AdditiveBlendAttitude", type = "bool", value = false },
      { name = "AdditiveBlendPosition", type = "bool", value = false },
      { name = "StartEventIndex", type = "int", value = 0, min = 0 },
      { name = "SphericallyInterpolateTrajectoryPosition", type = "bool", value = false },
      { name = "DurationEventBlendPassThrough", type = "bool", value = false},
      { name = "DurationEventBlendInSequence", type = "bool", value = true},
      { name = "DurationEventBlendSameUserData", type = "bool", value = false},
      { name = "DurationEventBlendOnOverlap", type = "bool", value = false},
      { name = "DurationEventBlendWithinRange", type = "bool", value = false},
      { name = "Source0BranchOptimization", type = "bool", value = true },
      { name = "Source1BranchOptimization", type = "bool", value = true },
      { name = "Source0IgnoreEventWeight", type = "bool", value = false },
      { name = "Source1IgnoreEventWeight", type = "bool", value = false },
      { name = "UpdateWeightingOnce", type = "bool", value = false},
      { name = "Source0IsFullBodyAnimation", type = "bool", value = true},
    },

    --------------------------------------------------------------------------------------------------------------------
    serialize = function(node, Stream)
      local node0 = -1
      local node1 = -1
      local weightNodeID = -1
      local eventWeightNodeID = -1
      local loop = getAttribute(node, "Loop")
      local useSource1Additively = getAttribute(node, "UseSource1Additively")
      local additiveAtt = getAttribute(node, "AdditiveBlendAttitude")
      local additivePos = getAttribute(node, "AdditiveBlendPosition")
      local startEventIndex = getAttribute(node, "StartEventIndex")
      local sphericalTrajPos = getAttribute(node, "SphericallyInterpolateTrajectoryPosition")
      local source0BranchOptimization = getAttribute(node, "Source0BranchOptimization")
      local source1BranchOptimization = getAttribute(node, "Source1BranchOptimization")
      local source0IgnoreEventWeight = getAttribute(node, "Source0IgnoreEventWeight")
      local source1IgnoreEventWeight = getAttribute(node, "Source1IgnoreEventWeight")
      local UpdateWeightingOnce = getAttribute(node, "UpdateWeightingOnce")
      local Source0IsFullBodyAnimation = getAttribute(node, "Source0IsFullBodyAnimation")

      if isConnected{SourcePin  =(node .. ".Source0"), ResolveReferences = true} then 
        node0 = getConnectedNodeID(node, "Source0")
      end
      if isConnected{SourcePin  =(node .. ".Source1"), ResolveReferences = true} then 
        node1 = getConnectedNodeID(node, "Source1")
      end
      if isConnected{SourcePin  =(node .. ".Weight"), ResolveReferences = true} then 
        weightNodeID= getConnectedNodeID( (node .. ".Weight") ) 
        if weightNodeID == nil then
          weightNodeID = -1
        end
      else
        weightNodeID = -1
      end
      if isConnected{SourcePin  =(node .. ".EventBlendingWeight"), ResolveReferences = true} then 
        eventWeightNodeID= getConnectedNodeID( (node .. ".EventBlendingWeight") ) 
        if eventWeightNodeID == nil then
          eventWeightNodeID = -1
        end
      else
        eventWeightNodeID = -1
      end

      local versionNumber = 4
      Stream:writeInt(versionNumber, "versionNumber")
      
      Stream:writeUInt(node0, "Source0ConnectedNodeID")
      Stream:writeUInt(node1, "Source1ConnectedNodeID")
      Stream:writeUInt(weightNodeID, "IDConnectedToWeight")
      Stream:writeUInt(eventWeightNodeID, "EventWeightNodeID")
      Stream:writeBool(loop, "Loop")
      Stream:writeBool(useSource1Additively, "UseSource1Additively")
      Stream:writeBool(additiveAtt, "AdditiveBlendAttitude")
      Stream:writeBool(additivePos, "AdditiveBlendPosition")
      Stream:writeInt(startEventIndex, "StartEventIndex")
      Stream:writeBool(sphericalTrajPos, "SphericallyInterpolateTrajectoryPosition")
      Stream:writeBool(source0BranchOptimization, "Source0BranchOptimization")
      Stream:writeBool(source1BranchOptimization, "Source1BranchOptimization")
      Stream:writeBool(source0IgnoreEventWeight, "Source0IgnoreEventWeight")
      Stream:writeBool(source1IgnoreEventWeight, "Source1IgnoreEventWeight")
      Stream:writeBool(UpdateWeightingOnce, "UpdateWeightingOnce")
     
      -- Duration event blending flags --
      local durationEventBlendPassThrough = getAttribute(node, "DurationEventBlendPassThrough")
      local durationEventBlendInSequence = getAttribute(node, "DurationEventBlendInSequence")
      local durationEventBlendSameUserData = getAttribute(node, "DurationEventBlendSameUserData")
      local durationEventBlendOnOverlap = getAttribute(node, "DurationEventBlendOnOverlap")
      local durationEventBlendWithinRange = getAttribute(node, "DurationEventBlendWithinRange")
      Stream:writeBool(durationEventBlendPassThrough, "DurationEventBlendPassThrough")
      Stream:writeBool(durationEventBlendInSequence, "DurationEventBlendInSequence")
      Stream:writeBool(durationEventBlendSameUserData, "DurationEventBlendSameUserData")
      Stream:writeBool(durationEventBlendOnOverlap, "DurationEventBlendOnOverlap")
      Stream:writeBool(durationEventBlendWithinRange, "DurationEventBlendWithinRange")
      
      Stream:writeBool(Source0IsFullBodyAnimation, "Source0IsFullBodyAnimation")
    end,

    --------------------------------------------------------------------------------------------------------------------
    validate = function(node)
      if isConnected{SourcePin  = (node..".Source0"), ResolveReferences = true} and isConnected{SourcePin  = (node..".Source1"), ResolveReferences = true} then
        local nodesConnectedTo0 = listConnections{Object = (node..".Source0"), ResolveReferences = true}
        local nodesConnectedTo1 = listConnections{Object = (node..".Source1"), ResolveReferences = true}
        local node0 = nodesConnectedTo0[1]
        local node1 = nodesConnectedTo1[1]
        if isValid(node0) ~= true or isValid(node1) ~= true then
          return nil, "Blend 2 Match Events requires two valid input nodes"
        end

        local type0 = getType(node0)
        local type1 = getType(node1)
        if (type0 == "AnimWithEvents" and type1 == "AnimWithEvents") then
          local looping = getAttribute(node, "Loop")
          if (looping) then
            local aEvents = getEventSequence(getAttribute(node0, "AnimationTake"))
            local bEvents = getEventSequence(getAttribute(node1, "AnimationTake"))
            if (table.getn(aEvents) > table.getn(bEvents)) then 
              if ((math.mod(table.getn(aEvents), table.getn(bEvents))) ~= 0) then
                return true
                --[[ NOTE JB: Want to return the following warning message with this result.
                              Should also be recursiveley checking if child nodes are looping.
                , ("Blend 2 Match Events node " .. node .. " has none wholly divisable source event sequences. A has " .. table.getn(aEvents) .. " events and B has " ..  table.getn(bEvents))]]--
              end
            else
              if (math.mod(table.getn(bEvents), table.getn(aEvents)) ~= 0) then
                return true
                --[[ NOTE JB: Want to return the following warning message with this result.
                              Should also be recursiveley checking if child nodes are looping.
                , ("Blend 2 Match Events node " .. node .. " has none wholly divisable source event sequences. A has " .. table.getn(aEvents) .. " events and B has " ..  table.getn(bEvents))]]--
              end
            end
          end
        end
      else 
        return nil, ("Blend 2 Match Events node " .. node .. " is missing a required connection to Source0 and/or Source1")
      end
      
      if isConnected{SourcePin  = (node..".Weight"), ResolveReferences = true} then
        local nodesConnectedToWeight = listConnections{Object = (node..".Weight"), ResolveReferences = true}
        local nodeWeight = nodesConnectedToWeight[1]
        if isValid(nodeWeight) ~= true then
	        return true, (node .. " has no valid input to weight pin, default to a value of 0")
        end
      else 
        return true, (node .. " has no input to weight pin, default to a value of 0")
      end
      
      return true
    end,

    --------------------------------------------------------------------------------------------------------------------
    getTransformChannels = function(node)
      local node0Channels = {} 
      local node1Channels = {} 
      if isConnected{SourcePin  = (node .. ".Source0"), ResolveReferences = true} then 
        local Source0Table = listConnections{Object = (node .. ".Source0"), ResolveReferences = true}
        local NodeConnectedTo0 = Source0Table[1]
        node0Channels = anim.getTransformChannels(NodeConnectedTo0)
      end

      if isConnected{SourcePin  = (node .. ".Source1"), ResolveReferences = true} then 
        local Source1Table = listConnections{Object = (node .. ".Source1"), ResolveReferences = true}
        local NodeConnectedTo1 = Source1Table[1]
        node1Channels = anim.getTransformChannels(NodeConnectedTo1)
      end

      local resultChannels = setUnion(node0Channels, node1Channels)
      return resultChannels
    end,

    getEvents = function(node)
      local _min = nil
      local _max = nil
      local nodeSources = { node .. ".Source0", node .. ".Source1" }

      for i, nodeSource in ipairs(nodeSources) do
        local connections = listConnections{Object = nodeSource, ResolveReferences = true}
        local connectedNode = connections[1]
        if connectedNode then
          local connectedEvents = anim.getEvents(connectedNode)

          if connectedEvents then
            if _min == nil or connectedEvents.min < _min then
              _min = connectedEvents.min
            end
            if _max == nil or connectedEvents.max > _max then
              _max = connectedEvents.max
            end
          end
        end
      end

      return { min = _min or 0, max = _max or 0 }
    end,

  }

)

------------------------------------------------------------------------------------------------------------------------
-- End of Blend2MatchEvents node definition.
------------------------------------------------------------------------------------------------------------------------


