registerNode("IOIAnim",
{
	helptext = "Supplies clip of animation data from an entry in the animation set",
	group = "Utilities",
	image = "IOIAnim.png",
	id = generateNamespacedId(idNamespaces.G2, G2NodeTypeNumbers.NodeTypeIOIAnimNum),
	--------------------------------------------------------------------------------------------------------------------
    
	functionPins = 
	{
		["Result"] = { 
						input = false,
						array = false,
						interfaces = {"Transforms", "Time", "Events"},
					},
	},
	
	attributes =
	{
		-- Attributes with the 'perAnimSet' flag set as true will have a widget per animation set,
		-- will generate data for each animation set.
		-- Note that the 'AnimationTake' type always has perAnimSet = true
		{ name = "AnimationTake", type = "animationTake", value = { filename = "", takename = "" } },
		{ name = "Loop", type = "bool", value = true },
		{ name = "PlayBackwards", type = "bool", value = false },
		{ name = "GenerateAnimationDeltas", type = "bool", value = false },
		{ name = "ClipStartFraction", type = "float", min = 0.0, max = 1.0, value = 0.0, perAnimSet = true },
		{ name = "ClipEndFraction", type = "float", min = 0.0, max = 1.0, value = 1.0, perAnimSet = true },
		{ name = "StartEventIndex", type = "int", value = 0, min = 0, perAnimSet = true },
	},
	
	--------------------------------------------------------------------------------------------------------------------
	serialize = function(node, Stream)
		-- Write a node version number as the first bit of data. This can then be used to make
		-- the node definition creator backwards compatible with earlier node versions. Thus we should be able
		-- submit new versions of our animation node without breaking the Gfx build if we keep the node definition
		-- creator backwards compatible.
		local versionNumber = 6
		Stream:writeInt(versionNumber, "versionNumber")
		
		local animSets = listAnimSets()
		local numAnimSets = table.getn(animSets)

		-- Parent NodeAnimWithEvents data:
		local animationTake = getAttribute(node, "AnimationTake")
		local animationIndex = animationTake.index
		local generateAnimationDeltas = getAttribute(node, "GenerateAnimationDeltas")      

		local loop = getAttribute(node, "Loop")
		local playBackwards = getAttribute(node, "PlayBackwards")

		Stream:writeInt(animationIndex, "AnimIndex")
		Stream:writeBool(generateAnimationDeltas, "GenerateAnimationDeltas")
		Stream:writeBool(loop, "Loop")
		Stream:writeBool(playBackwards, "PlayBackwards")
		Stream:writeUInt(numAnimSets, "NumAnimSets")

		for asIdx, asVal in animSets do
			local clipStartFraction = getAttribute(node, "ClipStartFraction", asVal)
			local ClipEndFraction = getAttribute(node, "ClipEndFraction", asVal)
			local startEventIndex = getAttribute(node, "StartEventIndex", asVal)

			-- Make sure the clip end time is greater or equal to the clip start time.
			if(ClipEndFraction < clipStartFraction) then
				ClipEndFraction = clipStartFraction
			end

			Stream:writeFloat(clipStartFraction, "ClipStartFraction")
			Stream:writeFloat(ClipEndFraction, "ClipEndFraction")
			Stream:writeInt(startEventIndex, "StartEventIndex")
		end
	end,
    
	--------------------------------------------------------------------------------------------------------------------
	validate = function(node)
		-- for now just a simple check to ensure that the file exists
		-- In the future validate against file types the runtime will support
		local valid = true
		local missingFiles = {} 
		local setsWithMissingFiles = {} 
		local errorMessage = ""

		local animSets = listAnimSets()
		local setsWithData = 0
		
		for setIdx, setName in animSets do
			local shouldCheckForData = hasAnimSetData( (node .. ".AnimationTake"), setName)
			
			if shouldCheckForData then
				local animationTake = getAttribute(node,"AnimationTake",setName)
				--  anim.takeExists will use the current best practice to determine if the take is present on disk.  The
				--  implementation might change in future version of connect to improve performance. 
				local validTake = anim.takeExists(animationTake)

				setsWithData = setsWithData + 1 

				if not validTake then
					local filename = "UNKNOWN"
					if animationTake then
						filename = animationTake.filename
					end
					valid = false
					table.insert(missingFiles, filename)
					table.insert(setsWithMissingFiles, setName)  
				end
			end
		end
		
		if valid then
			return true
		else

			-- the formatting of the error message depends if multiple sets are used.  If data is only in one set
			-- do not mention sets in the error message. 
			if(setsWithData == 1) then
				errorMessage = "The animation file referenced by "
				errorMessage = errorMessage .. node
				errorMessage = errorMessage .. " ["
				errorMessage = errorMessage .. missingFiles[1]
				errorMessage = errorMessage .. "] does not exist"
			else
				errorMessage = "The animation files referenced by "
				errorMessage = errorMessage .. node .. "  ["
				for idx, fileName in missingFiles do
					errorMessage = errorMessage .. setsWithMissingFiles[idx] .. " - " .. fileName .. "; "
				end
				errorMessage = errorMessage .. "] do not exist"
			end
			
			return nil, errorMessage 
		end
	end,
    
    --------------------------------------------------------------------------------------------------------------------
    getTransformChannels = function(node)
		local transformChannels = anim.getTransformChannels( (node .. ".AnimationTake") )
		return transformChannels
	end,

	--------------------------------------------------------------------------------------------------------------------
	getEvents = function(node)
		local animationTake = getAttribute(node .. ".AnimationTake")
		if animationTake then
			local tracks = anim.getTakeMarkupData(animationTake)
			if tracks then
				local _min = nil
				local _max = nil

				for i, track in pairs(tracks) do
					local numevents = table.getn(track)
					if _min == nil or numevents < _min then
						_min = numevents
					end
					
					if _max == nil or numevents > _max then
						_max = numevents
					end
				end
				return { min = _min or 0, max = _max or 0 }
			end
		end
		return { min = 0, max = 0 }
    end
}
)

registerAttributeDisplayInfo(
  "IOIAnim",

  {
    { title = "File",
      attributes = { "AnimationTake" },
      type = "AttributeWidget",
    },

    { title = "Clip Range",
      attributes = { "ClipStartFraction", "ClipEndFraction", "StartEventIndex" },
      type = "AttributeWidget",
    },

    { title = "Properties",
      attributes = { "Loop", "PlayBackwards", "GenerateAnimationDeltas" },
      type = "AttributeGroup",
    }
  }
)
