------------------------------------------------------------------------------------------------------------------------
-- IOI Delta Anim node definition.
------------------------------------------------------------------------------------------------------------------------

registerNode("IOIDeltaAnimMatchEvents",
  {
    helptext = "Subtracts Base transforms from Variation input and matches events",
    group = "Blends",
    image = "IOIDeltaAnimMatchEvents.png",
    id = generateNamespacedId(idNamespaces.IOI, IOINodeTypeNumbers.NodeTypeIOIDeltaAnimMatchEvents),

    --------------------------------------------------------------------------------------------------------------------
    functionPins = 
    {
      ["Variation"] = {
                       input = true, 
                       array = false,
                       interfaces = {"Transforms", "Time", "Events"},
                     },

      ["Base"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },

      ["Result"] = { 
                     input = false, 
                     array = false, 
                     interfaces = {"Transforms", "Time", "Events"}, 
                   },
    },

    dataPins = 
    {
       ["EventBlendingWeight"] = {
                      input = true, 
                      array = false, 
                      type = "float", 
                    },
    },

    pinOrder = {"Variation", "Base", "Result", "EventBlendingWeight"},

    --------------------------------------------------------------------------------------------------------------------
    attributes =
    {
      { name = "Loop", type = "bool", value = true },
      { name = "UseSource1Additively", type = "bool", value = true},
      { name = "AdditiveBlendAttitude", type = "bool", value = false },
      { name = "AdditiveBlendPosition", type = "bool", value = false },
      { name = "StartEventIndex", type = "int", value = 0, min = 0 },
      { name = "DurationEventBlendPassThrough", type = "bool", value = false},
      { name = "DurationEventBlendInSequence", type = "bool", value = true},
      { name = "DurationEventBlendSameUserData", type = "bool", value = false},
      { name = "DurationEventBlendOnOverlap", type = "bool", value = false},
      { name = "DurationEventBlendWithinRange", type = "bool", value = false},
    },

    --------------------------------------------------------------------------------------------------------------------
    serialize = function(node, Stream)
      local node0 = -1
      local node1 = -1
      local eventWeightNodeID = -1
      local loop = getAttribute(node, "Loop")
      local useSource1Additively = getAttribute(node, "UseSource1Additively")
      local additiveAtt = getAttribute(node, "AdditiveBlendAttitude")
      local additivePos = getAttribute(node, "AdditiveBlendPosition")
      local startEventIndex = getAttribute(node, "StartEventIndex")

      if isConnected{SourcePin  =(node .. ".Variation"), ResolveReferences = true} then 
        node0 = getConnectedNodeID(node, "Variation")
      end
      if isConnected{SourcePin  =(node .. ".Base"), ResolveReferences = true} then 
        node1 = getConnectedNodeID(node, "Base")
      end
      if isConnected{SourcePin  =(node .. ".EventBlendingWeight"), ResolveReferences = true} then 
        eventWeightNodeID= getConnectedNodeID( (node .. ".EventBlendingWeight") ) 
        if eventWeightNodeID == nil then
          eventWeightNodeID = -1
        end
      else
        eventWeightNodeID = -1
      end

      Stream:writeUInt(node0, "VariationConnectedNodeID")
      Stream:writeUInt(node1, "BaseConnectedNodeID")
      Stream:writeUInt(eventWeightNodeID, "EventWeightNodeID")
      Stream:writeBool(loop, "Loop")
      Stream:writeBool(useSource1Additively, "UseSource1Additively")
      Stream:writeBool(additiveAtt, "AdditiveBlendAttitude")
      Stream:writeBool(additivePos, "AdditiveBlendPosition")
      Stream:writeInt(startEventIndex, "StartEventIndex")
     
      -- Duration event blending flags --
      local durationEventBlendPassThrough = getAttribute(node, "DurationEventBlendPassThrough")
      local durationEventBlendInSequence = getAttribute(node, "DurationEventBlendInSequence")
      local durationEventBlendSameUserData = getAttribute(node, "DurationEventBlendSameUserData")
      local durationEventBlendOnOverlap = getAttribute(node, "DurationEventBlendOnOverlap")
      local durationEventBlendWithinRange = getAttribute(node, "DurationEventBlendWithinRange")
      Stream:writeBool(durationEventBlendPassThrough, "DurationEventBlendPassThrough")
      Stream:writeBool(durationEventBlendInSequence, "DurationEventBlendInSequence")
      Stream:writeBool(durationEventBlendSameUserData, "DurationEventBlendSameUserData")
      Stream:writeBool(durationEventBlendOnOverlap, "DurationEventBlendOnOverlap")
      Stream:writeBool(durationEventBlendWithinRange, "DurationEventBlendWithinRange")
    end,

    --------------------------------------------------------------------------------------------------------------------
    validate = function(node)
      if isConnected{SourcePin  = (node..".Variation"), ResolveReferences = true} and isConnected{SourcePin  = (node..".Base"), ResolveReferences = true} then
        local nodesConnectedTo0 = listConnections{Object = (node..".Variation"), ResolveReferences = true}
        local nodesConnectedTo1 = listConnections{Object = (node..".Base"), ResolveReferences = true}
        local node0 = nodesConnectedTo0[1]
        local node1 = nodesConnectedTo1[1]
        if isValid(node0) ~= true or isValid(node1) ~= true then
          return nil, "IOI Delta Anim Match Events requires two valid input nodes"
        end

        local type0 = getType(node0)
        local type1 = getType(node1)
        if (type0 == "AnimWithEvents" and type1 == "AnimWithEvents") then
          local looping = getAttribute(node, "Loop")
          if (looping) then
            local aEvents = getEventSequence(getAttribute(node0, "AnimationTake"))
            local bEvents = getEventSequence(getAttribute(node1, "AnimationTake"))
            if (table.getn(aEvents) > table.getn(bEvents)) then 
              if ((math.mod(table.getn(aEvents), table.getn(bEvents))) ~= 0) then
                return true
                --[[ NOTE JB: Want to return the following warning message with this result.
                              Should also be recursiveley checking if child nodes are looping.
                , ("IOI Delta Anim Match Events node " .. node .. " has none wholly divisable source event sequences. A has " .. table.getn(aEvents) .. " events and B has " ..  table.getn(bEvents))]]--
              end
            else
              if (math.mod(table.getn(bEvents), table.getn(aEvents)) ~= 0) then
                return true
                --[[ NOTE JB: Want to return the following warning message with this result.
                              Should also be recursiveley checking if child nodes are looping.
                , ("IOI Delta Anim Match Events node " .. node .. " has none wholly divisable source event sequences. A has " .. table.getn(aEvents) .. " events and B has " ..  table.getn(bEvents))]]--
              end
            end
          end
        end
      else 
        return nil, ("IOI Delta Anim Match Events node " .. node .. " is missing a required connection to Variation and/or Base")
      end
      
      return true
    end,

    --------------------------------------------------------------------------------------------------------------------
    getTransformChannels = function(node)
      local node0Channels = {} 
      local node1Channels = {} 
      if isConnected{SourcePin  = (node .. ".Variation"), ResolveReferences = true} then 
        local VariationTable = listConnections{Object = (node .. ".Variation"), ResolveReferences = true}
        local NodeConnectedTo0 = VariationTable[1]
        node0Channels = anim.getTransformChannels(NodeConnectedTo0)
      end

      if isConnected{SourcePin  = (node .. ".Base"), ResolveReferences = true} then 
        local BaseTable = listConnections{Object = (node .. ".Base"), ResolveReferences = true}
        local NodeConnectedTo1 = BaseTable[1]
        node1Channels = anim.getTransformChannels(NodeConnectedTo1)
      end

      local resultChannels = setUnion(node0Channels, node1Channels)
      return resultChannels
    end,

    getEvents = function(node)
      local _min = nil
      local _max = nil
      local nodeSources = { node .. ".Variation", node .. ".Base" }

      for i, nodeSource in ipairs(nodeSources) do
        local connections = listConnections{Object = nodeSource, ResolveReferences = true}
        local connectedNode = connections[1]
        if connectedNode then
          local connectedEvents = anim.getEvents(connectedNode)

          if connectedEvents then
            if _min == nil or connectedEvents.min < _min then
              _min = connectedEvents.min
            end
            if _max == nil or connectedEvents.max > _max then
              _max = connectedEvents.max
            end
          end
        end
      end

      return { min = _min or 0, max = _max or 0 }
    end,

  }

)

------------------------------------------------------------------------------------------------------------------------
-- End of IOI DeltaAnim node definition.
------------------------------------------------------------------------------------------------------------------------
