------------------------------------------------------------------------------------------------------------------------
-- IOI FilterTransforms node definition.
------------------------------------------------------------------------------------------------------------------------

registerFilterNode("IOIFilterTransforms",
  {
    helptext = "Filter out the specified channels from an input buffer. Pass through all other Node communication.",
    group = "Utilities",
    image = "IOIFilterTransforms.png",
    id = generateNamespacedId(idNamespaces.NaturalMotion, 105),
    
    interface = "Transforms",
    
    --------------------------------------------------------------------------------------------------------------------
    attributes =
    {
      { name = "ChannelIsOutput", type = "boolArray", perAnimSet = true, syncWithRigChannels = true },
    },
    
    --------------------------------------------------------------------------------------------------------------------
    serialize = function(node, Stream)
      local sourceID = -1
      local nodesConnectedTo
      local sourceNode

      if isConnected{SourcePin  = (node .. ".In"), ResolveReferences = true} then 
        sourceID = getConnectedNodeID(node, "In")
        nodesConnectedTo = listConnections{Object = (node .. ".In"), ResolveReferences = true}
        sourceNode = nodesConnectedTo[1]
      end  
      
      local animSets = listAnimSets()
      local numAnimSets = table.getn(animSets)

      Stream:writeUInt(sourceID, "NodeConnectedTo")
      Stream:writeUInt(numAnimSets, "NunAnimSets")

      for asIdx, asVal in animSets do
  
	    local channelIsOutput = getAttribute(node, "ChannelIsOutput", asVal)

        local notExportedChannelsList = {}
        for i,v in ipairs(channelIsOutput) do 
          if v == false then
            table.insert(notExportedChannelsList, i -1) 
          end
        end

        Stream:writeUInt(table.getn(notExportedChannelsList), "FilterIdCount")
        for i, v in ipairs(notExportedChannelsList) do 
          Stream:writeUInt(v, "Id")
        end
      
      end

    end,
    
    --------------------------------------------------------------------------------------------------------------------
    validate = function(node)
      if isConnected{SourcePin  = (node..".In"), ResolveReferences = true} then
        local nodesConnectedTo = listConnections{Object = (node..".In"), ResolveReferences = true}
        local sourceNode = nodesConnectedTo[1]
        if isValid(sourceNode) ~= true then
          return nil, "Filter Transforms requires a valid input node"
        end        
      else 
        return nil, ("Filter Transforms node " .. node .. " is missing a required connection to In")
      end
      return true
    end,
    
    --------------------------------------------------------------------------------------------------------------------
    getTransformChannels = function(node, set)
      if isConnected{SourcePin  = (node..".In"), ResolveReferences = true} then
        local nodesConnectedTo = listConnections{Object = (node .. ".In"), ResolveReferences = true}
        local sourceNode = nodesConnectedTo[1]
        local connectedChannels = anim.getTransformChannels(sourceNode, set)
        local exportedChannelsList = {}
        local arrayAttrValues = getAttribute(node, "ChannelIsOutput")
        for i,v in ipairs(arrayAttrValues) do 
           if v == true then
             table.insert(exportedChannelsList, i - 1)
           end
        end
        local resultChannels = setIntersection(exportedChannelsList, connectedChannels)
        return resultChannels
      else
        return {}
      end
    end,

    --------------------------------------------------------------------------------------------------------------------
    getEvents = function(node)
      if isConnected{SourcePin  = (node..".In"), ResolveReferences = true} then
        local nodesConnectedTo = listConnections{Object = (node .. ".In"), ResolveReferences = true}
        local sourceNode = nodesConnectedTo[1]
        local sourceEvents = anim.getEvents(sourceNode)
        return sourceEvents
      end
      
      return { min = 0, max = 0 }
    end,
    
    --------------------------------------------------------------------------------------------------------------------
    onAttributeInherited = function(nodeName, attributeName, setName)
      if attributeName == "ChannelIsOutput" then
        -- init filter channels with default anim set rig size, set all outputs 'true' by default.
        local numChannels = anim.getRigSize(setName)
        local channelIsOutputTable = { }
        for i = 1, numChannels do
          table.insert(channelIsOutputTable, true)
        end
        setAttribute(nodeName .. ".ChannelIsOutput", channelIsOutputTable, setName)
      end
    end,
  }
)

------------------------------------------------------------------------------------------------------------------------
-- End of IOI FilterTransforms node definition.
------------------------------------------------------------------------------------------------------------------------



------------------------------------------------------------------------------------------------------------------------
-- IOI FilterTransforms custom editor
------------------------------------------------------------------------------------------------------------------------
registerAttributeDisplayInfo(
  "IOIFilterTransforms",

  {
    { title = "Output Channel",
      attributes = { "ChannelIsOutput" },
      type = "ChannelNameWidget",
    },
  }
)
------------------------------------------------------------------------------------------------------------------------
-- End of IOI FilterTransforms custom editor
------------------------------------------------------------------------------------------------------------------------
