------------------------------------------------------------------------------------------------------------------------
-- Ground rotation node definition.
------------------------------------------------------------------------------------------------------------------------

registerNode("IOIGroundRotation",
  {
    helptext = "Modifies the ground, either replace translation and/or attitude or apply correction to translation and/or attitude.",
    group = "Utilities",
    image = "IOIGroundRotation.png", 
    id = generateNamespacedId(idNamespaces.G2, G2NodeTypeNumbers.NodeTypeIOIGroundRotation),
    
    --------------------------------------------------------------------------------------------------------------------
    functionPins = 
    {
		["Source"] = { input = true,	array = false, interfaces = {"Transforms", "Time", "Events"}, },
		["Result"] = { input = false,	array = false, interfaces = {"Transforms", "Time", "Events"}, },
    },
    
    --------------------------------------------------------------------------------------------------------------------
    dataPins = 
    {
		["Angle"] = { input = true,	array = false, type = "float" },
		["Trajectory"] = { input = true, array = false, type = "vector3" },
		["TranslationThreshold"] = { input = true,	array = false, type = "float" },
		["AttitudeThreshold"] = { input = true,	array = false, type = "float" },
	},
    
    pinOrder = {"Source", "Angle", "Trajectory", "TranslationThreshold", "AttitudeThreshold", "Result" },
    
    --------------------------------------------------------------------------------------------------------------------
    
    attributes =
    {
      { name = "OverrideTrajectory", type = "bool", value = false },
      { name = "CorrectiveTranslation", type = "bool", value = false },
      { name = "CorrectiveAttitude", type = "bool", value = false },
      { name = "RotateTrajectory", type = "bool", value = false },
      { name = "RotateTrajectoryToEndPosition", type = "bool", value = false },
      { name = "CorrectionStartEventID", type = "int", value = 300 },
      { name = "CorrectionEndEventID", type = "int", value = 301 },
      { name = "ApplyAsDelta", type = "bool", value = false},
      { name = "MirrorTrajectory", type = "bool", value = false},
    },
    
    
    --------------------------------------------------------------------------------------------------------------------
    serialize = function(node, Stream)
      local inputNode = -1
      local angleNode = -1
      local trajectoryNode = -1
      local translationThresholdNode = -1
      local attitudeThresholdNode = -1
      local bOverrideTrajectory = getAttribute(node, "OverrideTrajectory")
      local bCorrectiveTranslation = getAttribute(node, "CorrectiveTranslation")
      local bCorrectiveAttitude = getAttribute(node, "CorrectiveAttitude")
      local bRotateTrajectory = getAttribute(node, "RotateTrajectory")
      local bRotateTrajectoryToEndPosition = getAttribute(node, "RotateTrajectoryToEndPosition")
      local nCorrectionStartEventID = getAttribute(node, "CorrectionStartEventID")
      local nCorrectionEndEventID = getAttribute(node, "CorrectionEndEventID")
      local bApplyAsDelta = getAttribute(node, "ApplyAsDelta")
      local bMirrorTrajectory = getAttribute(node, "MirrorTrajectory")
      
      if isConnected{SourcePin  = (node .. ".Source"), ResolveReferences = true} then 
        inputNode = getConnectedNodeID(node, "Source")
      end
      
      if isConnected{SourcePin  = (node .. ".Angle"), ResolveReferences = true} then 
        angleNode = getConnectedNodeID(node, "Angle")
      end
      
      if isConnected{SourcePin = (node .. ".Trajectory"), ResolveReferences = true} then
		trajectoryNode = getConnectedNodeID(node, "Trajectory")
      end
      
      if isConnected{SourcePin  = (node .. ".TranslationThreshold"), ResolveReferences = true} then 
        translationThresholdNode = getConnectedNodeID(node, "TranslationThreshold")
      end

      if isConnected{SourcePin  = (node .. ".AttitudeThreshold"), ResolveReferences = true} then 
        attitudeThresholdNode = getConnectedNodeID(node, "AttitudeThreshold")
      end

      Stream:writeUInt(4, "Version")
      Stream:writeUInt(inputNode, "InputNodeID")
      Stream:writeUInt(angleNode, "AngleNodeID")
      Stream:writeUInt(trajectoryNode, "TrajectoryNodeID")
      Stream:writeUInt(translationThresholdNode, "TranslationThresholdNodeID")
      Stream:writeUInt(attitudeThresholdNode, "AttitudeThresholdNodeID")
      Stream:writeBool(bOverrideTrajectory, "OverrideTrajectory")
      Stream:writeBool(bCorrectiveTranslation, "CorrectiveTranslation")
      Stream:writeBool(bCorrectiveAttitude, "CorrectiveAttitude")
      Stream:writeBool(bRotateTrajectory, "RotateTrajectory")
      Stream:writeBool(bRotateTrajectoryToEndPosition, "RotateTrajectoryToEndPosition")
      Stream:writeUInt(nCorrectionStartEventID, "CorrectionStartEventID")
      Stream:writeUInt(nCorrectionEndEventID, "CorrectionEndEventID")
      Stream:writeBool(bApplyAsDelta, "ApplyAsDelta")
      Stream:writeBool(bMirrorTrajectory, "MirrorTrajectory")
      
    end,
    
    --------------------------------------------------------------------------------------------------------------------
    validate = function(node)
      if isConnected(node..".Source") then
        local nodesConnected = listConnections{Object = (node..".Source"), ResolveReferences = true}
        local inputNode = nodesConnected[1]
        if isValid(inputNode) ~= true then
          return nil, "IOIGroundRotation requires a valid input node"
        end
        
      else 
        return nil, ("IOIGroundRotation node " .. node .. " is missing a required connection to Source")
      end
      
      if isConnected(node..".Angle") then
        local nodesConnected = listConnections{Object = (node..".Angle"), ResolveReferences = true}
        local inputNode = nodesConnected[1]
        if isValid(inputNode) ~= true then
          return nil, "IOIGroundRotation requires a valid input node"
        end
        
      else 
        return nil, ("IOIGroundRotation node " .. node .. " is missing a required connection to Angle")
      end
      
      local bOverrideTrajectory = getAttribute(node, "OverrideTrajectory")
      if bOverrideTrajectory then
		if not isConnected{SourcePin = (node .. ".Trajectory"), ResolveReferences = true} then
			return nil, "Node "..node.." is set to override trajectory, but is missing a connection to that parameter."
		end
	  end
      
      local bCorrectiveTranslation = getAttribute(node, "CorrectiveTranslation")
      if bCorrectiveTranslation then
		if not isConnected{SourcePin = (node .. ".Trajectory"), ResolveReferences = true} then
			return nil, "Node "..node.." is set to correct trajectory, but is missing a connection to Trajectory parameter."
		end
	  end

      local bCorrectiveAttitude = getAttribute(node, "CorrectiveAttitude")
      if bCorrectiveAttitude then
		if not isConnected{SourcePin = (node .. ".Angle"), ResolveReferences = true} then
			return nil, "Node "..node.." is set to correct attitude, but is missing a connection to Angle parameter."
		end
	  end

	  return true
    end,
    
    --------------------------------------------------------------------------------------------------------------------
    getTransformChannels = function(node)
      local inputNodeChannels = {} 
      if isConnected((node .. ".Source")) then 
        local SourceTable = listConnections{Object = (node .. ".Source"), ResolveReferences = true}
        local NodeConnected = SourceTable[1]
        inputNodeChannels = anim.getTransformChannels(NodeConnected)
      end

      return inputNodeChannels
    end,

  }
)

------------------------------------------------------------------------------------------------------------------------
-- End of IOIGroundRotation node definition.
------------------------------------------------------------------------------------------------------------------------






