------------------------------------------------------------------------------------------------------------------------
-- IOIHeadController node definition.
------------------------------------------------------------------------------------------------------------------------

registerNode("IOIHeadController",
  {
    helptext = "IOIHeadController: Controls head and eye movement of character through IK and behaviour",
    group = "Utilities",
    image = "IOIHeadController.png",
    id = generateNamespacedId(idNamespaces.IOI, IOINodeTypeNumbers.NodeTypeIOIHeadController),
    
    --------------------------------------------------------------------------------------------------------------------
    functionPins = 
    {
      ["Source"] = {
                       input = true, 
                       array = false,
                       interfaces = {"Transforms", "Time"},
                     },
                     
      ["Result"] = { 
                     input = false, 
                     array = false, 
                     interfaces = {"Transforms", "Time"}, 
                   },
    },
    
    dataPins = 
    {
        ["Target"] = {
                      input = true, 
                      array = false, 
                      type = "vector3", 
                    },
        ["TargetNoise"] = {
                      input = true, 
                      array = false, 
                      type = "float", 
                    },
        ["TargetWeight"] = {
                      input = true, 
                      array = false, 
                      type = "float", 
                    },
        ["HeadBias"] = {
                      input = true, 
                      array = false, 
                      type = "float", 
                    },
        ["Drowsiness"] = {
                      input = true, 
                      array = false, 
                      type = "float", 
                    },
        ["BlendWeight"] = {
                      input = true, 
                      array = false, 
                      type = "float", 
                    },
    },
    
    pinOrder = {"Source", "Result", "Target", "TargetNoise", "TargetWeight", "HeadBias", "Drowsiness", "BlendWeight"},
    
    --------------------------------------------------------------------------------------------------------------------
    attributes =
    {
      { name = "HeadEndJoint", type = "int", value = 0, perAnimSet = true },
      { name = "HeadRootJoint", type = "int", value = 0, perAnimSet = true },
      { name = "HeadPointingVectorX", type = "float", value = 0 },
      { name = "HeadPointingVectorY", type = "float", value = 1 },
      { name = "HeadPointingVectorZ", type = "float", value = 0 },
      { name = "HeadEndEffectorOffsetX", type = "float", value = 0 },
      { name = "HeadEndEffectorOffsetY", type = "float", value = 0 },
      { name = "HeadEndEffectorOffsetZ", type = "float", value = 0 },
      { name = "LeftEyeJoint", type = "int", value = 0, perAnimSet = true },
      { name = "RightEyeJoint", type = "int", value = 0, perAnimSet = true },
      { name = "LeftEyelidJoint", type = "int", value = 0, perAnimSet = true },
      { name = "RightEyelidJoint", type = "int", value = 0, perAnimSet = true },
      { name = "LeftEyebrowJoint", type = "int", value = 0, perAnimSet = true },
      { name = "RightEyebrowJoint", type = "int", value = 0, perAnimSet = true },
      { name = "UpdateTargetByDeltas", type = "bool", value = false },
    },
    
    --------------------------------------------------------------------------------------------------------------------
    serialize = function(node, Stream)
      local inputNodeID = -1
      local targetNodeID = -1
      local targetNoiseNodeID = -1
      local targetWeightNodeID = -1
      local headBiasNodeID = -1
      local drowsinessNodeID = -1
      local blendWeightNodeID = -1
      
      if isConnected{SourcePin = (node .. ".Source") , ResolveReferences = true} then 
        inputNodeID = getConnectedNodeID(node, "Source")
      end
      
      if isConnected{SourcePin = (node .. ".Target") , ResolveReferences = true} then 
        targetNodeID = getConnectedNodeID(node, "Target")
      end
      
      if isConnected{SourcePin = (node .. ".TargetNoise") , ResolveReferences = true} then 
        targetNoiseNodeID = getConnectedNodeID(node, "TargetNoise")
      end

      if isConnected{SourcePin = (node .. ".TargetWeight") , ResolveReferences = true} then 
        targetWeightNodeID = getConnectedNodeID(node, "TargetWeight")
      end
      
      if isConnected{SourcePin = (node .. ".HeadBias") , ResolveReferences = true} then 
        headBiasNodeID = getConnectedNodeID(node, "HeadBias")
      end

      if isConnected{SourcePin = (node .. ".Drowsiness") , ResolveReferences = true} then 
        drowsinessNodeID = getConnectedNodeID(node, "Drowsiness")
      end

      if isConnected{SourcePin = (node .. ".BlendWeight") , ResolveReferences = true} then 
        blendWeightNodeID = getConnectedNodeID(node, "BlendWeight")
      end

      local animSets = listAnimSets()
      local numAnimSets = table.getn(animSets)

      Stream:writeUInt(numAnimSets, "NumAnimSets")
      
      Stream:writeUInt(inputNodeID, "InputNodeID")

      Stream:writeUInt(targetNodeID, "targetPosNodeID")
      Stream:writeUInt(targetNoiseNodeID, "targetNoiseNodeID")
      Stream:writeUInt(targetWeightNodeID, "targetWeightNodeID")
      Stream:writeUInt(headBiasNodeID, "headBiasNodeID")
      Stream:writeUInt(drowsinessNodeID, "drowsinessNodeID")
      Stream:writeUInt(blendWeightNodeID, "blendWeightNodeID")
      
      local pointingVectorX = getAttribute(node, "HeadPointingVectorX")
      Stream:writeFloat(pointingVectorX)
      local pointingVectorY = getAttribute(node, "HeadPointingVectorY")
      Stream:writeFloat(pointingVectorY)
      local pointingVectorZ = getAttribute(node, "HeadPointingVectorZ")
      Stream:writeFloat(pointingVectorZ)
      
      local offsetX = getAttribute(node, "HeadEndEffectorOffsetX")
      Stream:writeFloat(offsetX)
      local offsetY = getAttribute(node, "HeadEndEffectorOffsetY")
      Stream:writeFloat(offsetY)
      local offsetZ = getAttribute(node, "HeadEndEffectorOffsetZ")
      Stream:writeFloat(offsetZ)
      
      local updateTargetByDeltas = getAttribute(node, "UpdateTargetByDeltas")
      Stream:writeBool(updateTargetByDeltas)
      
      for asIdx, asVal in animSets do
        local index = getAttribute(node, "HeadEndJoint", asVal)
        Stream:writeUInt(index)
        index = getAttribute(node, "HeadRootJoint", asVal)
        Stream:writeUInt(index)

        index = getAttribute(node, "LeftEyeJoint", asVal)
        Stream:writeUInt(index)
        index = getAttribute(node, "RightEyeJoint", asVal)
        Stream:writeUInt(index)

        index = getAttribute(node, "LeftEyelidJoint", asVal)
        Stream:writeUInt(index)
        index = getAttribute(node, "RightEyelidJoint", asVal)
        Stream:writeUInt(index)

        index = getAttribute(node, "LeftEyebrowJoint", asVal)
        Stream:writeUInt(index)
        index = getAttribute(node, "RightEyebrowJoint", asVal)
        Stream:writeUInt(index)
      end

    end,
    
    --------------------------------------------------------------------------------------------------------------------
    validate = function(node)
    
      local inputNode
      
      -- Validate connections
      if isConnected{SourcePin = (node .. ".Source") , ResolveReferences = true} then
        local nodesConnected = listConnections{Object = (node .. ".Source") , ResolveReferences = true}
        inputNode = nodesConnected[1]
        if isValid(inputNode) ~= true then
          return nil, ("HeadLook node " .. node .. " requires a valid input node")
        end
        
      else 
        return nil, ("HeadLook node " .. node .. " is missing a required connection to Source")
      end
          
      -- Validate Rig indices
      local animSets = listAnimSets()
      local numAnimSets = table.getn(animSets)

      for asIdx, asVal in animSets do

        local rigSize = anim.getRigSize(asVal)

        local index = getAttribute(node, "LeftEyeJoint", asVal)
        if index == nil or index <= 0 or index >= rigSize then
          return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. ") requires a valid LeftEyeJoint")
        end
        index = getAttribute(node, "RightEyeJoint", asVal)
        if index == nil or index <= 0 or index >= rigSize then
          return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. ") requires a valid RightEyeJoint")
        end
        index = getAttribute(node, "LeftEyelidJoint", asVal)
        if index == nil or index <= 0 or index >= rigSize then
          return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. ") requires a valid LeftEyelidJoint")
        end
        index = getAttribute(node, "RightEyelidJoint", asVal)
        if index == nil or index <= 0 or index >= rigSize then
          return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. ") requires a valid RightEyelidJoint")
        end
        index = getAttribute(node, "LeftEyebrowJoint", asVal)
        if index == nil or index <= 0 or index >= rigSize then
          return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. ") requires a valid LeftEyebrowJoint")
        end
        index = getAttribute(node, "RightEyebrowJoint", asVal)
        if index == nil or index <= 0 or index >= rigSize then
          return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. ") requires a valid RightEyebrowJoint")
        end

        index = getAttribute(node, "HeadEndJoint", asVal)
        local rootIndex = getAttribute(node, "HeadRootJoint", asVal)
        if index == nil or index <= 0 or index >= rigSize then
          return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. ") requires a valid HeadEndJoint")
        end
        if rootIndex == nil or rootIndex <= 0 or rootIndex >= rigSize then
          return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. ") requires a valid HeadRootJoint")
        end
        
        local connectedChannels = anim.getTransformChannels(inputNode, asIdx)
        while index ~= rootIndex do
          if index == nil or index <= 0 or index >= rigSize then
            return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. "): the root joint is not an ancestor of the end joint")
          end
          
          -- see if the channel is being ouput
          local foundChannel = false
          for channelIdx, channelVal in connectedChannels do      
            if channelVal == index then
              foundChannel = true
            end
          end
          
          if not foundChannel then
            return nil, ("HeadLook node " .. node .. " (animset " .. asVal .. "): the IK bones are not being output by the input node.")
          end
          
          index = anim.getParentBoneIndex(index, asVal)
        end
      end

      if isConnected{SourcePin = (node .. ".Target") , ResolveReferences = true} then
        local nodesConnected = listConnections{Object = (node .. ".Target") , ResolveReferences = true}
        local inputNode = nodesConnected[1]
        if isValid(inputNode) ~= true then
          return nil, ("HeadLook node " .. node .. " requires a valid Look-At Target")
        end
        
      else 
        return nil, ("HeadLook node " .. node .. " is missing a required connection to Target")
      end

      if isConnected{SourcePin = (node .. ".TargetNoise") , ResolveReferences = true} then
        local nodesConnected = listConnections{Object = (node .. ".TargetNoise") , ResolveReferences = true}
        local inputNode = nodesConnected[1]
        if isValid(inputNode) ~= true then
          return nil, ("HeadLook node " .. node .. " requires a valid Target Noise")
        end
      end

      if isConnected{SourcePin = (node .. ".TargetWeight") , ResolveReferences = true} then
        local nodesConnected = listConnections{Object = (node .. ".TargetWeight") , ResolveReferences = true}
        local inputNode = nodesConnected[1]
        if isValid(inputNode) ~= true then
          return nil, ("HeadLook node " .. node .. " requires a valid Target Weight")
        end
      end

      if isConnected{SourcePin = (node .. ".HeadBias") , ResolveReferences = true} then
        local nodesConnected = listConnections{Object = (node .. ".HeadBias") , ResolveReferences = true}
        local inputNode = nodesConnected[1]
        if isValid(inputNode) ~= true then
          return nil, ("HeadLook node " .. node .. " requires a valid Head Bias")
        end
      end

      if isConnected{SourcePin = (node .. ".Drowsiness") , ResolveReferences = true} then
        local nodesConnected = listConnections{Object = (node .. ".Drowsiness") , ResolveReferences = true}
        local inputNode = nodesConnected[1]
        if isValid(inputNode) ~= true then
          return nil, ("HeadLook node " .. node .. " requires a valid Drowsiness")
        end
      end

      if isConnected{SourcePin = (node .. ".BlendWeight") , ResolveReferences = true} then
        local nodesConnected = listConnections{Object = (node .. ".BlendWeight") , ResolveReferences = true}
        local inputNode = nodesConnected[1]
        if isValid(inputNode) ~= true then
          return nil, ("HeadLook node " .. node .. " requires a valid Blend Weight")
        end
      end

      return true
    end,
    
    --------------------------------------------------------------------------------------------------------------------
    getTransformChannels = function(node)
      local inputNodeChannels = {} 
      if isConnected{SourcePin = (node .. ".Source"), ResolveReferences = true} then 
        local SourceTable = listConnections{Object = (node .. ".Source") , ResolveReferences = true}
        local NodeConnected = SourceTable[1]
        inputNodeChannels = anim.getTransformChannels(NodeConnected)
      end

      return inputNodeChannels
    end,

  }

)

------------------------------------------------------------------------------------------------------------------------
-- End of IOIHeadController node definition.
------------------------------------------------------------------------------------------------------------------------







