------------------------------------------------------------------------------------------------------------------------
-- BlendNMatchEvents node definition.
-------------------------------------------------------------------------------------------------------------------------

registerNode("IOIParametricBlendNMatchEvents",
  {
    helptext = "Blends N animation streams together and time warps to match the synchronisation events, weights are defined by animation's speed and (in future) other parameters",
    group = "Blends",
    image = "BlendNMatchEvents.png",
    id = generateNamespacedId(idNamespaces.IOI, IOINodeTypeNumbers.NodeTypeIOIParametricBlendNMatchEvents),

    --------------------------------------------------------------------------------------------------------------------
    functionPins = 
    {
      ["Source0"] = { 
                       input = true, 
                       array = false,
                       interfaces = {"Transforms", "Time", "Events"},
                     },
                     
      ["Source1"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },
  
      ["Source2"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },
                     
      ["Source3"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },
                     
      ["Source4"] = {
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },
                     
      ["Source5"] = {
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },
                     
      ["Source6"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },
                     
      ["Source7"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },
                     
      ["Source8"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },
      
      ["Source9"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },
      ["Source10"] = { 
                       input = true, 
                       array = false, 
                       interfaces = {"Transforms", "Time", "Events"}, 
                     },           
      ["Result"] = { 
                     input = false, 
                     array = false, 
                     interfaces = {"Transforms", "Time", "Events"}, 
                   },
    },
    
    dataPins = 
    {
       ["Weight"] = {
                      input = true, 
                      array = false, 
                      type = "float", 
                    },
    },
    
    pinOrder =
    {
      "Source0",
      "Source1",
      "Source2",
      "Source3",
      "Source4",
      "Source5",
      "Source6",
      "Source7",
      "Source8",
      "Source9",
      "Source10",
      "Weight",
      "Result"
    },
    
    --------------------------------------------------------------------------------------------------------------------
    attributes =
    {
      { name = "Loop", type = "bool", value = true },
      { name = "StartEventIndex", type = "int", value = 0, min = 0 },
      { name = "SphericallyInterpolateTrajectoryPosition", type = "bool", value = false },
      { name = "DurationEventBlendPassThrough",type = "bool", value = false},
      { name = "DurationEventBlendInSequence",type = "bool", value = true},
      { name = "DurationEventBlendSameUserData",type = "bool", value = false},
      { name = "DurationEventBlendOnOverlap",type = "bool", value = false},
      { name = "DurationEventBlendWithinRange",type = "bool", value = false},
      { name = "BranchOptimization", type = "bool", value = true },
      { name = "Source0IgnoreEventWeight", type = "bool", value = false },
      { name = "Source1IgnoreEventWeight", type = "bool", value = false },
      { name = "UpdateWeightingOnce", type = "bool", value = false },
    },
    
    --------------------------------------------------------------------------------------------------------------------
    serialize = function(node, Stream)
      local connectedNodeCount = 0
      local connectedPinCount = 0
      local connectedIds = {} 
      local pinWeight = {} 
      local startEventIndex = getAttribute(node, "StartEventIndex")

      for currentPin = 0, 10 do
        local pinName = (node .. ".Source" .. currentPin)
        if isConnected{SourcePin  = pinName, ResolveReferences = true} then 
          connectedPinCount = connectedPinCount + 1
          connectedIds[currentPin] = getConnectedNodeID(pinName)
        else 
          break
        end
      end
      
      local versionNumber = 1
      Stream:writeInt(versionNumber, "versionNumber")
      
      -- do we want this nodes output to loop. 
      local loop = getAttribute(node, "Loop")
      Stream:writeBool(loop, "Loop")   

      -- Which event to start playback of this blend on.
      Stream:writeInt(startEventIndex, "StartEventIndex")
      
      -- do we want this node to spherically interpolate the input delta trajectory translations.
      local sphericalTrajPos = getAttribute(node, "SphericallyInterpolateTrajectoryPosition")
      Stream:writeBool(sphericalTrajPos, "SphericallyInterpolateTrajectoryPosition")
      
	  -- do we want this node to perform blend tree optimizations?
      local branchOptimization = getAttribute(node, "BranchOptimization")
      Stream:writeBool(branchOptimization, "BranchOptimization")
      
      -- Do we want to ignore blend weights when sampling events?
      local source0IgnoreEventWeight = getAttribute(node, "Source0IgnoreEventWeight")
      local source1IgnoreEventWeight = getAttribute(node, "Source1IgnoreEventWeight")
      Stream:writeBool(source0IgnoreEventWeight, "Source0IgnoreEventWeight")
      Stream:writeBool(source1IgnoreEventWeight, "Source1IgnoreEventWeight")
      
      local UpdateWeightingOnce = getAttribute(node, "UpdateWeightingOnce")
      Stream:writeBool(UpdateWeightingOnce, "UpdateWeightingOnce")
      
      -- Duration event blending flags --
      local durationEventBlendPassThrough = getAttribute(node, "DurationEventBlendPassThrough")
      local durationEventBlendInSequence = getAttribute(node, "DurationEventBlendInSequence")
      local durationEventBlendSameUserData = getAttribute(node, "DurationEventBlendSameUserData")
      local durationEventBlendOnOverlap = getAttribute(node, "DurationEventBlendOnOverlap")
      local durationEventBlendWithinRange = getAttribute(node, "DurationEventBlendWithinRange")
      Stream:writeBool(durationEventBlendPassThrough, "DurationEventBlendPassThrough")
      Stream:writeBool(durationEventBlendInSequence, "DurationEventBlendInSequence")
      Stream:writeBool(durationEventBlendSameUserData, "DurationEventBlendSameUserData")
      Stream:writeBool(durationEventBlendOnOverlap, "DurationEventBlendOnOverlap")
      Stream:writeBool(durationEventBlendWithinRange, "DurationEventBlendWithinRange")

      -- next write out the id of the node connected to the weight parameter
      if isConnected{SourcePin  =  (node .. ".Weight"), ResolveReferences = true } then 
        local wId = getConnectedNodeID( (node .. ".Weight") ) 
        if wId ~= nil then
          Stream:writeUInt(wId, "IDConnectedToWeight")
        else
          Stream:writeUInt(-1, "ConnectedToInvalidNode")
        end
      else
        Stream:writeUInt(-1, "ConnectedToInvalidNode") -- treat 0xFFFFFFFF as invalid
      end

      -- now write out the number of connected pins
      Stream:writeInt(connectedPinCount)

      -- finally write connectedPinCount pairs of runtime id and pin weight
      for currentPin = 0, (connectedPinCount - 1) do 
        Stream:writeInt(connectedIds[currentPin], "ConnectedNodeID")
      end        
    end,
    
    --------------------------------------------------------------------------------------------------------------------
    validate = function(node)
      local connectedPinCount = 0
      local pinsAllowed = true
      local error = false
      local warn = false
      local errorMessage = ""
      local warnMessage = ""
      
      for currentPin = 0, 10 do
        local pinName = (node .. ".Source" .. currentPin)
        if isConnected{SourcePin  = pinName, ResolveReferences = true} then 
          if pinsAllowed then
           connectedPinCount = connectedPinCount + 1
          else
           error = true
           errorMessage = "The pins on Blend N Match Events node " .. node .. " are not connected consecutively. Ensure that there is not a gap between connections" 
          end
          
          local nodesConnected = listConnections{Object = pinName, ResolveReferences = true}
          if isValid(nodesConnected[1]) ~= true then
           error = true
           errorMessage = "The pins on Blend N Match Events node " .. node .. " are not connected to valid nodes" 
          end
        else 
          pinsAllowed = false
        end
      end
      
      if not(error or warn) then 
        if connectedPinCount < 2 then
          error = true
          errorMessage = "The Blend N Match Events node " .. node .. " requires at least two connections"
        end
      end

      if not(error or warn) then
      -- Check that connected pins modulo division of number of events in animation sources if looping.
      -- Not an error, but will not produce a nice looping blend.
        local looping = getAttribute(node, "Loop")
        if (looping) then
          local eventCount = nil
          for currentPin = 0, 10 do
            if isConnected{SourcePin  = (node..".Source"..currentPin), ResolveReferences = true} then
              local connectedNodes = listConnections{Object = (node..".Source"..currentPin), ResolveReferences = true}
              local connectedNode = connectedNodes[1]
              local connectedNodeType = getType(connectedNode)
              if connectedNodeType == "AnimWithEvents" then
                local eventSequence = getEventSequence(getAttribute(connectedNode, "AnimationTake"))
                if not eventCount then
                  eventCount = table.getn(eventSequence)
                else
                  if (eventCount > table.getn(eventSequence)) then 
                    if ((math.mod(eventCount, table.getn(eventSequence))) ~= 0) then
                      --[[ NOTE JB: Want to return the following warning message with this result.
                                    Should also be recursiveley checking if child nodes are looping.
                      warningMessage = "Blend N Match Events node " .. node .. " has none wholly divisable source event sequences."]]--
                    end
                  else
                    if (math.mod(table.getn(eventSequence), eventCount) ~= 0) then
                      --[[ NOTE JB: Want to return the following warning message with this result.
                                    Should also be recursiveley checking if child nodes are looping.
                      warningMessage = "Blend N Match Events node " .. node .. " has none wholly divisable source event sequences."]]--
                    end
                  end                  
                end
              end
            end
          end
        end
      end

      if isConnected{SourcePin  = (node..".Weight"), ResolveReferences = true} then
        local nodesConnectedToWeight = listConnections{Object = (node..".Weight"), ResolveReferences = true}
        local nodeWeight = nodesConnectedToWeight[1]
        if isValid(nodeWeight) ~= true then
			    warn = true
			    warnMessage = node .. " has no valid input to weight pin, default to a value of 0"
        end
      else 
		    warn = true
		    warnMessage = node .. " has no input to weight pin, default to a value of 0"
      end
      
      -- verify that the node is correctly setup
      if error then
        return nil, errorMessage
      end
      if warn then
		    return true, warnMessage
      end
      return true
    end,

    --------------------------------------------------------------------------------------------------------------------
    getTransformChannels = function(node)
      local unionOfResults = {} 
      local firstPin = true
      for currentPin = 0, 10 do
        local pinName = (node .. ".Source" .. currentPin)
        if isConnected{SourcePin  = pinName, ResolveReferences = true} then 
          local sourceTable = listConnections{Object =  pinName, ResolveReferences = true}
          local sourceNode = sourceTable[1]
          local curChannels = anim.getTransformChannels(sourceNode)
          if firstPin then
            unionOfResults = curChannels
          else
            unionOfResults = setUnion(unionOfResults, curChannels)
          end
        else 
          break
        end
      end
      return unionOfResults
    end,

    --------------------------------------------------------------------------------------------------------------------
    getEvents = function(node)
      local _min = nil
      local _max = nil

      for currentPin = 0, 10 do
        local pinName = (node .. ".Source" .. currentPin)
        if isConnected{SourcePin  = pinName, ResolveReferences = true} then 
          local sourceTable = listConnections{Object = pinName, ResolveReferences = true}
          local sourceNode = sourceTable[1]
          local sourceEvents = anim.getEvents(sourceNode)

          if sourceEvents then
            if _min == nil or sourceEvents.min < _min then
              _min = sourceEvents.min
            end
            if _max == nil or sourceEvents.max > _max then
              _max = sourceEvents.max
            end
          end

        end
      end

      return { min = _min or 0, max = _max or 0 }
    end,
    
  }
)

------------------------------------------------------------------------------------------------------------------------
-- End of BlendNMatchEvents node definition.
------------------------------------------------------------------------------------------------------------------------







