------------------------------------------------------------------------------------------------------------------------
-- IOISelectiveFeatherBlend node definition.
------------------------------------------------------------------------------------------------------------------------

registerNode("IOISelectiveFeatherBlend",
	{
		helptext = "Blends two animation streams together. The Blend Weight determines the respective contribution from Source0 and Source1 to the final transform buffer. Feather Weight determines the amount of feather blending performed on Source1.",
		group = "Blends",
		image = "IOISelectiveFeatherBlend.png",
		id = generateNamespacedId(idNamespaces.G2, G2NodeTypeNumbers.NodeTypeIOISelectiveFeatherBlend),
		
		--------------------------------------------------------------------------------------------------------------------
		functionPins = 
		{
			["Source0"] = {
				input = true, 
				array = false,
				interfaces = {"Transforms", "Time"},
				},
                     
			["Source1"] = { 
				input = true, 
				array = false, 
				interfaces = {"Transforms", "Time"}, 
				},

			["Result"] = { 
				input = false, 
				array = false, 
				interfaces = {"Transforms", "Time"}, 
				},
		},
    
		dataPins = 
		{
			["BlendWeight"] = {
				input = true, 
				array = false, 
				type = "float", 
			},
		
			["FeatherWeight"] = {
				input = true, 
				array = false, 
				type = "float", 
				},

			["IgnoreTrajectory0"] = {
				input = true, 
				array = false, 
				type = "float", 
				},

			["IgnoreTrajectory1"] = {
				input = true, 
				array = false, 
				type = "float", 
				},
		},
    
		pinOrder = {"Source0", "Source1", "Result", "BlendWeight", "FeatherWeight", "IgnoreTrajectory0", "IgnoreTrajectory1"},
    
		--------------------------------------------------------------------------------------------------------------------
		attributes =
		{
			{ name = "ChannelWeights", type = "floatArray", min = 0, max = 1, perAnimSet = true, syncWithRigChannels = true },
			{ name = "Loop", type = "bool", value = true },
			{ name = "UseSource1Additively", type = "bool", value = true},
			{ name = "AdditiveBlendAttitude", type = "bool", value = false },
			{ name = "AdditiveBlendPosition", type = "bool", value = false },
			{ name = "SphericallyInterpolateTrajectoryPosition", type = "bool", value = false },
			{ name = "Source0BranchOptimization", type = "bool", value = true },
			{ name = "Source1BranchOptimization", type = "bool", value = true },
			{ name = "Source0IgnoreEventWeight", type = "bool", value = false },
			{ name = "Source1IgnoreEventWeight", type = "bool", value = false },
			{ name = "UpdateWeightingOnce", type = "bool", value = false},
		},
    
		--------------------------------------------------------------------------------------------------------------------
		serialize = function(node, Stream)
			local node0 = -1
			local node1 = -1
			local weightNodeID = -1
			local featherWeightNodeID = -1
			local ignoreTrajectory0NodeID = -1
			local ignoreTrajectory1NodeID = -1
			local loop = getAttribute(node, "Loop")
			local useSource1Additively = getAttribute(node, "UseSource1Additively")
			local additiveAtt = getAttribute(node, "AdditiveBlendAttitude")
			local additivePos = getAttribute(node, "AdditiveBlendPosition")
			local sphericalTrajPos = getAttribute(node, "SphericallyInterpolateTrajectoryPosition")
			local source0BranchOptimization = getAttribute(node, "Source0BranchOptimization")
			local source1BranchOptimization = getAttribute(node, "Source1BranchOptimization")
			local source0IgnoreEventWeight = getAttribute(node, "Source0IgnoreEventWeight")
			local source1IgnoreEventWeight = getAttribute(node, "Source1IgnoreEventWeight")			
			local UpdateWeightingOnce = getAttribute(node, "UpdateWeightingOnce")

			if(isConnected{SourcePin  = (node .. ".Source0") , ResolveReferences = true}) then 
				node0 = getConnectedNodeID(node, "Source0")
			end
			if(isConnected{SourcePin  = (node .. ".Source1"), ResolveReferences = true}) then 
				node1 = getConnectedNodeID(node, "Source1")
			end
			if(isConnected{SourcePin  = (node .. ".BlendWeight"), ResolveReferences = true}) then 
				weightNodeID = getConnectedNodeID(node, "BlendWeight")
			end
			if(isConnected{SourcePin = (node .. ".FeatherWeight"), ResolveReferences = true}) then
				featherWeightNodeID = getConnectedNodeID(node, "FeatherWeight")
			end
			if(isConnected{SourcePin = (node .. ".IgnoreTrajectory0"), ResolveReferences = true}) then
				ignoreTrajectory0NodeID = getConnectedNodeID(node, "IgnoreTrajectory0")
			end
			if(isConnected{SourcePin = (node .. ".IgnoreTrajectory1"), ResolveReferences = true}) then
				ignoreTrajectory1NodeID = getConnectedNodeID(node, "IgnoreTrajectory1")
			end
			
			-- Write version number for version control
			Stream:writeInt(4, "SelectiveFeatherBlendNodeVersion");
			
			-- Write feather blend properties
			Stream:writeUInt(featherWeightNodeID, "IDConnectedToFeatherWeight")

			-- Write ignore trajectory properties
			Stream:writeUInt(ignoreTrajectory0NodeID, "IDConnectedToIgnoreTrajectory0")
			Stream:writeUInt(ignoreTrajectory1NodeID, "IDConnectedToIgnoreTrajectory1")
 
			local ChannelWeightsValues = getAttribute(node, "ChannelWeights", 1)
			local numChannelWeightValues = table.getn(ChannelWeightsValues) -- this might be unsafe if the array contains nil values.

			Stream:writeUInt(numChannelWeightValues, "numChannelWeightValues")
			for i,v in ipairs(ChannelWeightsValues) do 
				Stream:writeFloat(v, "ChannelWeight")
			end
			
			-- Blend2 properties
			Stream:writeUInt(node0, "Source0ConnectedNodeID")
			Stream:writeUInt(node1, "Source1ConnectedNodeID")
			Stream:writeUInt(weightNodeID, "IDConnectedToWeight")
			Stream:writeBool(loop, "Loop")
			Stream:writeBool(useSource1Additively, "Usesource1Additively")
			Stream:writeBool(additiveAtt, "AdditiveBlendAttitude")
			Stream:writeBool(additivePos, "AdditiveBlendPosition")
			Stream:writeBool(sphericalTrajPos, "SphericallyInterpolateTrajectoryPosition")
			Stream:writeBool(source0BranchOptimization, "Source0BranchOptimization")
			Stream:writeBool(source1BranchOptimization, "Source1BranchOptimization")
			Stream:writeBool(source0IgnoreEventWeight, "Source0IgnoreEventWeight")
			Stream:writeBool(source1IgnoreEventWeight, "Source1IgnoreEventWeight")
			Stream:writeBool(UpdateWeightingOnce, "UpdateWeightingOnce")
		end,
    
		--------------------------------------------------------------------------------------------------------------------
		validate = function(node)
			
			-- Validate source input connections
			if(not isConnected{SourcePin  = (node..".Source0"), ResolveReferences = true}) then
				return nil, string.format("Selective Feather Blend node %s has no input connected at source0.", node)
			end
			if(not isConnected{SourcePin  = (node..".Source1"), ResolveReferences = true}) then
				return nil, string.format("Selective Feather Blend node %s has no input connected at source1.", node)
			end
			
			-- Validate connected input nodes
			local nodesConnectedTo0 = listConnections{Object = (node..".Source0"), ResolveReferences = true}
			local nodesConnectedTo1 = listConnections{Object = (node..".Source1"), ResolveReferences = true}
			local sourceNode0 = nodesConnectedTo0[1]
			local sourceNode1 = nodesConnectedTo1[1]
			if(not isValid(sourceNode0)) then
				return nil, string.format("%s has an invalid source node %s at input 0", node, sourceNode0.name)
			end
			if(not isValid(sourceNode1)) then
				return nil, string.format("%s has an invalid source node %s at input 1", node, sourceNode1.name)
			end
			
			-- Validate FeatherWeight input value
			if(not isConnected{SourcePin  = (node..".FeatherWeight"), ResolveReferences = true}) then
				return true, string.format("%s has no input to feather weight pin, defaulting to a value of 0.0", node)
			else
				local nodesConnectedToFeatherWeight = listConnections{Object = (node..".FeatherWeight"), ResolveReferences = true}
				local nodeFeatherWeight = nodesConnectedToFeatherWeight[1]
				if(not isValid(nodeFeatherWeight)) then
					return true, string.format("%s has no valid input to feather weight pin, default to a value of 0.0", node)
				end
			end

			return true
		end,
    
		--------------------------------------------------------------------------------------------------------------------
		getTransformChannels = function(node)
			local source0Channels = {} 
			local source1Channels = {} 
			if(isConnected{SourcePin  = (node .. ".Source0"), ResolveReferences = true}) then 
				local Source0NodeTable = listConnections{Object = (node .. ".Source0") , ResolveReferences = true}
				source0Channels = anim.getTransformChannels(Source0NodeTable[1])
			end

			if(isConnected{SourcePin  = (node .. ".Source1"), ResolveReferences = true}) then 
				local Source1NodeTable = listConnections{Object = (node .. ".Source1") , ResolveReferences = true}
				source1Channels = anim.getTransformChannels(Source1NodeTable[1])
			end

			local resultChannels = setUnion(source0Channels, source1Channels)
			return resultChannels
		end,
		
		getEvents = function(node)
			local _min = nil
			local _max = nil
			local nodeSources = { node .. ".Source0", node .. ".Source1" }

			for i, nodeSource in ipairs(nodeSources) do
				local connections = listConnections{Object = nodeSource, ResolveReferences = true}
				local connectedNode = connections[1]
				if(connectedNode ~= nil) then
					local connectedEvents = anim.getEvents(connectedNode)
					if(connectedEvents) then
						if(_min == nil or connectedEvents.min < _min) then
							_min = connectedEvents.min
						end
						if(_max == nil or connectedEvents.max > _max) then
							_max = connectedEvents.max
						end
					end
				end
			end
			return { min = _min or 0, max = _max or 0 }
		end,

		--------------------------------------------------------------------------------------------------------------------
		onAttributeInherited = function(nodeName, attributeName, setName)
			if attributeName == "ChannelWeights" then
				-- init filter channels with default anim set rig size, set all outputs 'true' by default.
				local numChannels = anim.getRigSize(setName)
				local channelIsOutputTable = { }
				for i = 1, numChannels do
					table.insert(channelIsOutputTable, 1)
				end
				setAttribute(nodeName .. ".ChannelWeights", channelIsOutputTable, setName)
			end
		end,
	}
)

------------------------------------------------------------------------------------------------------------------------
-- End of IOISelectiveFeatherBlend node definition.
------------------------------------------------------------------------------------------------------------------------






