-- --
-- Source control integration using the G2 ResourceServerCLI interface.
-- --

local sourceControlStatusTable = {}

local SCVerbose = false -- Set this to true to get SC print outs

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
-- PRIVATE FUNCTIONS
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 

local function SCPrint(string)
	if(SCVerbose) then
		app.log(string)
	end
end

local function addCachedSCStatus(filePath, scStatus, timeOfRequestToServer)
	local cachedStatusTable = {}
	cachedStatusTable["time"] = timeOfRequestToServer
	cachedStatusTable["value"] = scStatus
	
	sourceControlStatusTable[filePath] = cachedStatusTable
end

local function checkAgeOfCachedSCStatus(filePath)
	local cachedStatusTable = sourceControlStatusTable[filePath]
	if(cachedStatusTable ~= nil) then
		if(os.time() - cachedStatusTable["time"] > 1) then
			-- Cached value is timed out.
			SCPrint(string.format("TIMED OUT cached result for: '%s' (FilePath -> SCStatus)", filePath))
			sourceControlStatusTable[filePath] = nil
		end
	end
end

local function getCachedSCStatus(filePath)
	local cachedStatusTable = sourceControlStatusTable[filePath]
	if(cachedStatusTable ~= nil) then
		local result = cachedStatusTable["value"]
		SCPrint(string.format("Got cached result: '%s' -> '%s' (FilePath -> SCStatus)", filePath, tostring(result)))
		return result
	end
	
	return nil
end

local function purgeCachedSCStatus(filePath)
	SCPrint(string.format("PURGED cached result for: '%s' (FilePath -> SCStatus)", filePath))
	sourceControlStatusTable[filePath] = nil
end


-- --
-- Gets the source control status of a file.
-- --
local function scGetStatus(filePath)
	checkAgeOfCachedSCStatus(filePath)
	
	local result = getCachedSCStatus(filePath)
	if(result ~= nil) then
		return result
	end

	SCPrint("scGetStatus(" .. filePath .. ")")
	local timeOfRequestToServer = os.time()
	local res = runResourceServerCommand1Arg("GetResourceSourceControlStatus", filePath)
	local error = nil
	local result = {}
	
	if res then
		for line in io.lines(PFfile) do
			for k, v in string.gfind(line, "(%w+)=(%w+)") do
				if(tostring(v) == "True") then
					result[k] = true
				elseif(tostring(v) == "False") then
					result[k] = false
				else
					result[k] = v
				end
				SCPrint(" ADDING " .. tostring(k) .. " value " .. tostring(v) )
			end
		end
	else
		error = "FAIL"
	end
	addCachedSCStatus(filePath, result, timeOfRequestToServer)

	--[[
	SCPrint("-- RAW DATA --")
	SCPrint(sourceControlInfo)

	SCPrint("-- PAIRS --")
	for k,v in pairs(result) do
		SCPrint("  "..tostring(k)..": "..tostring(v))

	end
	SCPrint("-----------")
	--]]
	
	return result, error
end

local function scAdd(filePath)
	SCPrint("scAdd(" .. filePath .. ")")
	
	purgeCachedSCStatus(filePath)

	return runResourceServerCommand1Arg("AddResourceToSourceControl", filePath)
end

local function scCheckOut(filePath)
	SCPrint("scCheckOut(" .. filePath .. ")")
	
	purgeCachedSCStatus(filePath)

	return runResourceServerCommand1Arg("CheckOut", filePath)
end

local function scLock(filePath)
	SCPrint("scLock(" .. filePath .. ")")

	purgeCachedSCStatus(filePath)

	return runResourceServerCommand1Arg("Lock", filePath)
end

local function scUnlock(filePath)
	SCPrint("scUnlock(" .. filePath .. ")")

	purgeCachedSCStatus(filePath)

	return runResourceServerCommand1Arg("Unlock", filePath)
end

local function scGetLatest(filePath)
	SCPrint("scGetLatest("..filePath..")")
	
	purgeCachedSCStatus(filePath)

	return runResourceServerCommand1Arg("GetLatest", filePath)
end

-- --
-- Check out a file in the default changelist.
-- Asks if it should get latest if a newer version exists.
-- Returns whether the file is in Perforce and checked out. Will return true if the file is already checked out.
-- --
local function EnsureFileIsCheckedOut(filePath, lockFile)
	if(lockFile == nil) then
		-- Default to not locking the file.
		lockFile = false
	end
	
	SCPrint("EnsureFileIsCheckedOut(" .. filePath .. ", " .. tostring(lockFile) .. ")")

	-- Check whether the specified file is in the depot
	local isInDepot, errMsg = IsFileInDepot(filePath)
	if(errMsg ~= nil) then
		ReportError(errMsg)
		return false
	end

	if(isInDepot) then
		-- If the file is not in the users changelist already, we need to check it out
		isInChangeList, errMsg = IsFileInOurChangeList(filePath)
		if(errMsg ~= nil) then
			ReportError(errMsg)
			return false
		end

		if(isInChangeList) then
			return true
		end
		
		local isCheckedOut = CheckOutFile(filePath, lockFile)
		return isCheckedOut
	end
end

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
-- PUBLIC FUNCTIONS
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 

-- --
-- Asks to open a file for add if it is not already in the users change list
-- --
function OpenForAdd(fileName, forceAdd)
	local addDialogue = true
	
	-- We assume the file is not already under source control, so if it is not in the clients change list, it must be opened for add
	local result, errMsg = IsFileInOurChangeList(fileName)
	if(errMsg ~= nil) then
		ReportError(errMsg)
		return false
	else
		if (not result) then
			if (not forceAdd) then
				addDialogue = ShowDialogue("Add to Source Control", string.format("File:\n\n%s\nis not under source control.\n\nDo you wish to add it?\n\n", fileName), BUTTON_YES_NO)
			end
			
			if (addDialogue) then
				local added, errMsg = scAdd(fileName)
				if (not added) then
					ShowDialogue("Add to source control", string.format("Could not add %s to source control!\n\n Output:\n%s\n\n", fileName, errorMsg), BUTTON_OK_ONLY)
					return false
				end
			end
		end
	end
	
	return true
end

-- --
-- Check out a file in the default changelist.
-- Asks if it should get latest if a newer version exists.
-- Requires the file to be in Perforce.
-- --
function CheckOutFile(filePath, lockFile)
	if(lockFile == nil) then
		-- Default to not locking the file.
		lockFile = false
	end

	SCPrint("CheckOutFile(" .. filePath .. ", " .. tostring(lockFile) .. ")")


	local sourceControlStatus, errorMessage = scGetStatus(filePath)
	if(sourceControlStatus == nil) then
		if(errorMessage ~= nil) then
			ReportError(errorMessage)
		else
			ReportError("Unknown error during check out from source control!")
		end
		return false
	end
	
	if(not sourceControlStatus["valid"]) then
		ReportError("Error: Can't check out file that is not in source control!")
		return false
	end
	
	-- If the file is not in the users changelist already, we need to check it out
	result, errMsg = IsFileInOurChangeList(filePath)
	if(errMsg ~= nil) then
		ReportError(errMsg)
		return false
	end
	

	if(not result) then
		result, errMsg  = IsFileLockedByOther(filePath)
		if(errMsg ~= nil) then
			ReportError(errMsg)
			return false
		end
		if(result) then
			-- Target file is checked out by someone else, we cannot check out.
			ShowDialogue("File is Locked", string.format("File %s\n is locked in source control.\n\nCannot check out.\n", filePath), BUTTON_OK_ONLY)
			return false		
		else
			-- Check for newer revisions of the file.
			result, errMsg = IsFileUpToDate(filePath)
			if(errMsg ~= nil) then
				ReportError(errMsg)
				return false
			end
			
			if(not result) then
				-- There is a newer revision available. Ask the user if he wants it.
				result = ShowDialogue("Newer Version Available", string.format("There is a newer revision available of the file:\n%s\nDo you want to get the latest revision before cheking out the file?\n", filePath), BUTTON_YES_NO)
				if(result) then
					result, errMsg = GetLatestRevision(filePath)
					if(errMsg ~= nil) then
						ReportError(errMsg)
						return false
					end
				end
			end			

			-- Check out the target file under the default changelist
			wasCheckedOut = scCheckOut(filePath)
			if (wasCheckedOut) then            
				-- Lock the file if requested
				if (lockFile) then
					scLock(filePath)
				end						
			else
				ShowDialogue("Failed Check Out", string.format("Failed to check out file:\n %s\n\n", filePath), BUTTON_OK_ONLY)
				return false
			end
		end
	else
		-- Check if the file is opened for integration in which case it will still be write protected.
		result, errMsg = IsFileOpenForIntegration(filePath)
		if(errMsg ~= nil) then
			ReportError(errMsg)
			return false
		else
			if(result) then
				ShowDialogue("Check Out File", string.format("File %s is open for integration.\n\nCannot open for edit.\nUse \"re-open for edit\" in your Perforce client.\n\n", filePath), BUTTON_OK_ONLY)
				return false
			end
		end
	end
	
	-- File is checked out
	return true
end

-- --
-- Lock a file in the depot
-- --
function LockFile(filePath)
	SCPrint("LockFile(" .. filePath .. ")")
	local sourceControlStatus, errorMessage = scGetStatus(filePath)
	if(sourceControlStatus ~= nil) then
		if(sourceControlStatus["valid"]) then
			return scLock(filePath)
		else
			return false, errorMessage
		end
	else
		return nil, errorMessage
	end
end

-- --
-- Checks if a given file exists in the depot
-- --
function IsFileInDepot(filePath)
	SCPrint("IsFileInDepot(" .. filePath .. ")")
	local sourceControlStatus, errorMessage = scGetStatus(filePath)
	if(sourceControlStatus ~= nil) then
		return sourceControlStatus["valid"] and (sourceControlStatus["headAction"] ~= "delete")
	else
		return nil, errorMessage
	end
end
-- --
-- Checks if a given file is already in the clients change list
-- --
function IsFileInOurChangeList(filePath)
	SCPrint("IsFileInOurChangeList(" .. filePath .. ")")
	local sourceControlStatus, errorMessage = scGetStatus(filePath)
	if(sourceControlStatus ~= nil) then
		return sourceControlStatus["inOurChangeList"]
	else
		return nil, errorMessage
	end
end

-- --
-- Checks if a given file is in the some clients change list
-- --
function IsFileInChangeList(filePath)
	SCPrint("IsFileInChangeList(" .. filePath .. ")")
	local sourceControlStatus, errorMessage = scGetStatus(filePath)
	if(sourceControlStatus ~= nil) then
		return sourceControlStatus["inChangeList"]
	else
		return nil, errorMessage
	end
end

-- --
-- Checks if a given file is open for integration and thus not writable
-- --
function IsFileOpenForIntegration(filePath)
	SCPrint("IsFileOpenForIntegration(" .. filePath .. ")")
	local sourceControlStatus, errorMessage = scGetStatus(filePath)
	if(sourceControlStatus ~= nil) then
		return sourceControlStatus["action"] == "integrate"
	else
		return nil, errorMessage
	end
end

-- --
-- Checks if a given file is locked by another user
-- --
function IsFileLockedByOther(filePath)
	SCPrint("IsFileLockedByOther(" .. filePath .. ")")
	local sourceControlStatus, errorMessage = scGetStatus(filePath)
	if(sourceControlStatus ~= nil) then
		return sourceControlStatus["locked"] and not sourceControlStatus["ourLock"]
	else
		return nil, errorMessage
	end
end
-- --
-- Checks if the client has the most recent revision of a given file
-- --
function IsFileUpToDate(filePath)
	SCPrint("IsFileUpToDate("..filePath..")")
	local sourceControlStatus, errorMessage = scGetStatus(filePath)
	if(sourceControlStatus ~= nil) then
		return sourceControlStatus["haveRevision"] == sourceControlStatus["headRevision"]
	else
		return nil, errorMessage
	end
end

-- --
-- Syncs a file to the latest revision in the depot
-- --
function GetLatestRevision(filePath)
	SCPrint("GetLatestRevision(" .. filePath .. ")")
	local sourceControlStatus, errorMessage = scGetStatus(filePath)
	if(sourceControlStatus ~= nil) then
		if(sourceControlStatus["valid"]) then
			return scGetLatest(filePath)
		else
			return false, errorMessage
		end
	else
		return nil, errorMessage
	end
end


-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
-- REGISTERED EVENT HANDLER FUNCTIONS
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 

-- --
-- Called when the user opens an mcn file
-- --
local function onOpenMcnFile(fileName)
	SCPrint("---------------------------------------")
	SCPrint("onOpenMcnFile(" .. fileName .. ")")
	
	SetCheckOutButtonEnable(false)
	
	local macroizedPath = utils.macroizeString(fileName)
	if(macroizedPath == fileName) then
		ShowDialogue("MC WARNING", string.format("Warning: You are opening the file\n%s\n\nwhich is outside your currently selected project folder:\n%s\n\nThis is likely to cause unexpected behaviour.\n\n", fileName, rootDir), BUTTON_OK_ONLY)
	end
	
	-- Check whether the specified file is in the depot
	local result, errMsg = IsFileInDepot(fileName)
	if(result == nil) then
		ReportError(errMsg)
		return
	end
	
	if (result) then
		SCPrint(" is in depot\n")

		-- If the file is in the depot, we need to check if it is already opened by this client
		result, errMsg = IsFileInOurChangeList(fileName)
		if(errMsg ~= nil) then
			ReportError(errMsg)
			return
		end
		
		if (not result) then
			SCPrint("  is not checked out by the current client\n")

			-- File is under source control. Check out?
			local CheckedOut = false
			local checkOutDlg = ShowDialogue("Open for Edit", string.format("File:\n\n%s\nis under source control.\n\nDo you wish to check it out?\n\n", fileName), BUTTON_YES_NO)
			if (checkOutDlg) then
				local doCheckOut = true
				local lockFile = true				
				
				-- Warn if the file is opened/locked by other client(s)
				result, errMsg = IsFileLockedByOther(fileName)
				if(errMsg ~= nil) then
					ReportError(errMsg)
					return
				end
				if (result) then
					ShowDialogue("Open for Edit", string.format("File:\n\n%s\nis exclusively locked. You will not be able to check it out before the lock is released.\n\n", fileName), BUTTON_OK_ONLY)
					checkOutDlg = false
				end
				
				if (not checkOutDlg) then
					doCheckOut = false
				else
					local lockFile = false
				end
				
				-- Check out the file
				if (doCheckOut) then
					
					-- If we want to check out the file, we need to check wether we have the latest revision.
					result, errMsg = IsFileUpToDate(fileName)
					if(errMsg ~= nil) then
						ReportError(errMsg)
						return
					end
					if (not result) then
						local getLatestDlg = ShowDialogue("File out of Date", string.format("A newer version of the file:\n\n%s\n is available. Do you wish to get the latest revision before checking out the file?\n\n", fileName), BUTTON_YES_NO)
						if (getLatestDlg) then
							scGetLatest(fileName)
						end
					end
					
					-- Check out the network file
					local errorMsg = ""
					CheckedOut, errorMsg = scCheckOut(fileName)
					if (CheckedOut) then            
						-- Lock the file if requested
						if (lockFile) then
							scLock(fileName)
						end						
					else
						ShowDialogue("Open for Edit", string.format("Could not open %s for edit!\n\n Output:\n%s\n\n", fileName, errorMsg), BUTTON_OK_ONLY)
					end
				end				
			end
			
			if (not CheckedOut) then
				ShowDialogue("Open for Edit","File was not checked out. You will not be able to save your changes.\n\n", BUTTON_OK_ONLY)
				SetCheckOutButtonEnable(true)
			end
		end
						
	else
		-- File is not under source control. Open for add?
		SCPrint("  is not in depot\n")
		OpenForAdd(fileName, false)
		
		-- If the network was opened for add, we create a preview/export custom data file to go with it
		if(IsFileInOurChangeList(fileName)) then
			SavePreviewRig()
			SaveExportRig()
			--AddExportRigFileToDepot(fileName)
		end
	end

	return CheckedOut
end


local function onSaveMcnFile(fileName)  
	SCPrint("---------------------------------------")
	SCPrint("onSaveMcnFile(" .. fileName .. ")")
	
	-- When saving a file, check if it is in the depot. If not, ask whether the user would like to add it.
	local result, errMsg = IsFileInDepot(fileName)
	if(errMsg ~= nil) then
		ReportError(errMsg)
	else
		if (not result) then
			OpenForAdd(fileName, false)
		end
		
		SavePreviewRig()
		SaveExportRig()
	end
end


local function onCloseMcnFile(fileName)
	SCPrint("---------------------------------------")
	SCPrint("onCloseMcnFile(" .. fileName .. ")")
end


-- Register event handlers:
registerEventHandler("mcFileOpenBegin", onOpenMcnFile)
registerEventHandler("mcFileSaveAsEnd", onSaveMcnFile)
registerEventHandler("mcFileSaveEnd", onSaveMcnFile)
registerEventHandler("mcFileCloseEnd", onCloseMcnFile)