-- G2 Custom interface functions
 local uiToolBarButtons = {}
 

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- PRIVATE FUNCTIONS
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

local function CloseRigPreferences()
	local dlg = ui.getWindow("RigPreferences")
	dlg:hide()
end

local function IsMcnOnDisk()
	return (mcn.filename() ~= "")
end

local function SaveRigPreferences()	
	local closePreferencesDialog = true
	
	-- Preview rig
	local fileControl = ui.getWindow("RigPreferences|PreviewRigFileNameControl")
	local newPreviewRigPath = fileControl:getValue()
	
	if(newPreviewRigPath ~= GetPreviewRigPath()) then
		local success, errMsg = SetPreviewRig(newPreviewRigPath)
		if(not success) then
			ReportError(string.format("Failed to set preview rig: %s", tostring(errMsg)));
			closePreferencesDialog = false
		end
		
		if(IsMcnOnDisk()) then
			local success, errMsg = SavePreviewRigPath(newPreviewRigPath)
			if(not success) then
				ReportError(string.format("Failed to save preview rig: %s", tostring(errMsg)));
				closePreferencesDialog = false
			end
		end
	end
	
	-- Export rig
	fileControl = ui.getWindow("RigPreferences|ExportRigFileNameControl")
	local newExportRigPath = fileControl:getValue()	

	if(newExportRigPath ~= GetExportRigPath()) then
		local success, errMsg = SetExportRig(newExportRigPath)
		if(not success) then
			ReportError(string.format("Failed to set export rig: %s", tostring(errMsg)));
			closePreferencesDialog = false
		end
		
		if(IsMcnOnDisk()) then
			local success, errMsg = SaveExportRigPath(newExportRigPath)
			if(not success) then
				ReportError(string.format("Failed to save export rig: %s\nIf you haven't already, you should check out the mcn file from within Morpheme:connect.", tostring(errMsg)));
				closePreferencesDialog = false
			end
		end
	end
	
	-- Only close the rig dialogue if the saving went well
	if(closePreferencesDialog) then
		CloseRigPreferences()
	end
end

local function SetRigPreferences()
	
	local setPreviewValue = true
	local setExportValue = true
	local previewRigPath, exportRigPath = GetRigPaths()
	if(previewRigPath == "") then
		setPreviewValue = false
		previewRigPath = string.format("%s\\Geometry\\Characters", assemblyPath)
	else
		previewRigPath = utils.demacroizeString(previewRigPath)
	end
		
	if(exportRigPath == "") then
		setExportValue = false
		exportRigPath = string.format("%s\\Geometry\\Characters", assemblyPath)
	else
		exportRigPath = utils.demacroizeString(exportRigPath)
	end
	
	local dlg = ui.getWindow("RigPreferences")
	if(not dlg)then		
		dlg = ui.createModalDialog{name = "RigPreferences", caption = "Rig Preferences", resize = false}
		dlg:beginVSizer()
			dlg:beginHSizer()
				dlg:addStaticText{text = "Preview Rig:"}
				dlg:addStretchSpacer{proportion = 1}
				dlg:addFilenameControl{	name = "PreviewRigFileNameControl",
										directory = previewRigPath,
										wildcard  = "XML files|XML",
										flags     = "expand",
										size = {width = 500}
										}
			dlg:endSizer()
			dlg:beginHSizer()
				dlg:addStaticText{text = "Export Rig:"}
				dlg:addStretchSpacer{proportion = 1}
				dlg:addFilenameControl{	name = "ExportRigFileNameControl",
										directory = exportRigPath,
										wildcard  = "XML files|XML",
										flags     = "expand",
										size = {width = 500}
										}
			dlg:endSizer()
			dlg:beginHSizer{flags = "right"}
				dlg:addButton{label = "OK", onClick = SaveRigPreferences}
				dlg:addButton{label = "Cancel", onClick = CloseRigPreferences}
			dlg:endSizer()
		dlg:endSizer()
	end
	
	if(setPreviewValue) then
		ui.getWindow("RigPreferences|PreviewRigFileNameControl"):setValue(previewRigPath)
	end
	if(setExportValue) then
		ui.getWindow("RigPreferences|ExportRigFileNameControl"):setValue(exportRigPath)
	end
	
	dlg:show()
end

local function AddRigSetupMenuItem()
	local editMenu = ui.getWindow("MainFrame|MainMenu|EditMenu")
	editMenu:addItem{ label = "Rig Setup...", accelerator = nil, name = "RigSetup", onClick = SetRigPreferences }
end

local function AddCharTypeToList(_charType, _list)
	local entryString = _charType.name
	
	-- Add four spaces for each level of depth to simulate a tree view via indentation
	for i = 0, _charType.depth*4, 1 do
		entryString = string.format(" %s", entryString)
	end

	local index = _list:getNumItems()+1
	_list:insertItem(index, entryString)
	
	-- Set the currently selected character type in bold
	local curCharType = GetCurrentCharType()
	if(_charType.name == curCharType.name) then
		_list:setItemBold(index, true)
	end
	
	-- Add any children to the list
	for i = 1, table.getn(_charType.children), 1 do
		AddCharTypeToList(_charType.children[i], _list)
	end	
end

local function RefreshCharTypeList()
	local list = ui.getWindow("CharTypeSelection|CharTypeList")
	list:removeAllItems()
	
	-- Retrieve a list of character types and put them on the list in order
	local charTypes = GetCharTypes()
	for i = 1, table.getn(charTypes), 1 do
		AddCharTypeToList(charTypes[i], list)
	end
end

local function CloseCharacterTypeDialogue()
	local dlg = ui.getWindow("CharTypeSelection")
	dlg:hide()
end

local function SetCharacterType()
	local list = ui.getWindow("CharTypeSelection|CharTypeList")
	local selectedIndex = list:getSelectedIndex()
	if(selectedIndex > 0) then
		local charTypeName = list:getItem(selectedIndex)
		
		-- Removed any preceeding spaces from the entry name
		local startIndex = string.find(charTypeName, "%a") -- Find the index of the first character
		local charTypeName = string.sub(charTypeName, startIndex)
		SetCurrentCharType(charTypeName)
		if(mcn.filename() ~= "") then
			SaveCharacterType()
		end
	end
	CloseCharacterTypeDialogue()
end

local function OpenCharacterTypeDialogue()
	local dlg = ui.getWindow("CharTypeSelection")
	if(not dlg)then		
		dlg = ui.createModalDialog{name = "CharTypeSelection", caption = "Select Variation", size = {width = 400, height = 200}, resize = true}
		
		-- Add a character type list to the dialogue
		dlg:addListControl{name = "CharTypeList", flags = "expand", proportion = 1}
		local list = ui.getWindow("CharTypeSelection|CharTypeList")
		list:setOnItemActivated(SetCharacterType)
		
		-- Add Open/Close buttons
		dlg:beginHSizer{flags = "right"}
			dlg:addButton{label = "OK", onClick = SetCharacterType}
			dlg:addButton{label = "Cancel", onClick = CloseCharacterTypeDialogue}
		dlg:endSizer()
	end
	
	RefreshCharTypeList()
	dlg:show()
end

local function AddCharTypeMenuItem()
	local editMenu = ui.getWindow("MainFrame|MainMenu|EditMenu")
	editMenu:addItem{ label = "Animation set variation...", accelerator = nil, name = "CharTypeSetup", onClick = OpenCharacterTypeDialogue }
end

local function CloseAnimDataBaseDialogue()
	local dlg = ui.getWindow("AnimDataBaseSetupDialogue")
	dlg:hide()
end

local function SetSelectedAnimDataBase()
	
	local fileControl = ui.getWindow("AnimDataBaseSetupDialogue|AnimDataBaseFileNameControl")
	local newAnimDataBasePath = fileControl:getValue()
	
	if(SetAnimDataBase(utils.macroizeString(newAnimDataBasePath))) then
		SetDefaultCharacterType()
		if(mcn.filename() ~= nil) then
			SaveAnimDataBasePath()
		end	
		CloseAnimDataBaseDialogue()
	end
end

local function OpenAnimDataBaseDialogue()	
	local animDBPath = GetCurrentAnimDBPath()
	if(animDBPath == "") then
		animDBPath = string.format("%s\\Animations\\AnimationDataBases", assemblyPath)
	end	
	animDBPath = utils.demacroizeString(animDBPath)
	
	local dlg = ui.getWindow("AnimDataBaseSetupDialogue")
	if(not dlg)then		
		dlg = ui.createModalDialog{name = "AnimDataBaseSetupDialogue", caption = "Select animation set", resize = false}
		dlg:beginVSizer()
			dlg:beginHSizer()
				dlg:addStaticText{text = "Animation set:"}
				dlg:addStretchSpacer{proportion = 1}
				dlg:addFilenameControl{	name = "AnimDataBaseFileNameControl",
										directory = animDBPath,
										wildcard  = "Anim set files|aset",	-- Changed from "Anim DB files|animDB"
										flags     = "expand",
										size = {width = 500}
										}
			dlg:endSizer()
			dlg:beginHSizer{flags = "right"}
				dlg:addButton{label = "OK", onClick = SetSelectedAnimDataBase}
				dlg:addButton{label = "Cancel", onClick = CloseAnimDataBaseDialogue}
			dlg:endSizer()
		dlg:endSizer()
	end
	
	dlg:show()
end

local function AddAnimDataBaseMenuItem()
	local editMenu = ui.getWindow("MainFrame|MainMenu|EditMenu")
	editMenu:addItem{ label = "Animation set...", accelerator = nil, name = "AnimDataBaseSetup", onClick = OpenAnimDataBaseDialogue }
end

local function onDropTextAndLocation(dropString, locationXY)
	-- Extract drop location coordinates from locationXY string
	local lastCharPos = getLastCharPos(locationXY, "|")
	local screenX = tonumber(string.sub(locationXY, 1, lastCharPos - 1))
	local screenY = tonumber(string.sub(locationXY, lastCharPos + 1))
	
	-- Create a Variation node at the specified location
	ParentPath = utils.getCurrentGraph()
	graphX, graphY = mcn.screenToGraphPos(ParentPath, screenX, screenY) -- Convert drop target coords to graph coords
	newVariationNodeName = create("IOIVariation", ParentPath, dropString)
	setAttribute(newVariationNodeName..".VariationName", dropString)
	setNodePosition(newVariationNodeName, graphX, graphY)
	
	-- Commit changes
	mcn.commit("onDropText")
end

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- PUBLIC FUNCTIONS
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

-- --
-- Adds a button tuple to the too bar button queue. These are then created in userInitStaticUI.
-- --
function AddUIButton(newButton)
	table.insert(uiToolBarButtons, newButton)
end

-- --
-- Called from NM scripts once standard UI has been created. Sets up custom tool bar interface buttons.
-- --
function userInitStaticUI()
	local toolBar = ui.getWindow("MainFrame|ToolBar")
	if(toolBar ~= nil) then
		toolBar:addSeparator()
		
		-- Create buttons in the same order they were added to the button table
		while (uiToolBarButtons[1] ~= nil) do
			toolBar:addButton{name = uiToolBarButtons[1].name, image = ui.createImage(uiToolBarButtons[1].imagePath), onClick = uiToolBarButtons[1].funcPointer, helpText = uiToolBarButtons[1].helpText}
			-- Set the enabled flag on the newly created button
			local newButton = ui.getWindow("MainFrame|ToolBar|"..uiToolBarButtons[1].name)
			if (newButton ~= nil) then
				newButton:enable(uiToolBarButtons[1].enabled)
			else
				ReportError("Failed to create custom UI button: "..uiToolBarButtons[1].name)
			end
			table.remove(uiToolBarButtons, 1)
		end
	else
		app.log("Failed to get mainframe tool bar!\n")
	end
	
	-- Grey out the default animation set button
	local animSetMenu = ui.getWindow("MainFrame|MainMenu|EditMenu|AnimationSets")
	if(animSetMenu == nil) then
		app.log("Animation Sets menu entry not found!")
	else
		animSetMenu:enable(false)
	end

	local editMenu = ui.getWindow("MainFrame|MainMenu|EditMenu")
	editMenu:addSeparator()
	AddRigSetupMenuItem()
	AddAnimDataBaseMenuItem()
	AddCharTypeMenuItem()
end

-- -- 
-- The code here is a temporary fix from Natural Motion. Normally only custom code should be placed here.
-- --
function userInitDocumentUI()
	-- Add calls here to initialise per document user interface components
	local layoutManager = ui.getWindow("MainFrame|LayoutManager")
	addBoneLookupWindow(layoutManager)
end

-- -- 
-- Network validation dialogue lifted from Morpheme 2.x. In 1.3.8 it is exposed in their scripts.
-- --
function showNetworkValidationReport(ids, warnings, errors)
	-- used to store a list of object paths associated with the warnings and errors
	local objects = nil

	local dlg = ui.createModelessDialog{caption = "Network Validation Report"}
	dlg:beginVSizer{}
		local listCtrl = dlg:addListControl{
			flags = "expand", proportion = 1, size = { width = 500, height = 200 },
			numColumns = 1,
			onItemActivated = function(listControl, rowIndex)
				if objects[rowIndex] ~= nil and objects[rowIndex] ~= "" then
					select(objects[rowIndex])
				end
			end
		}
		dlg:beginHSizer{flags = "expand", proportion = 0}
		dlg:addStretchSpacer{proportion = 1};
		dlg:addButton{
			label = "Ok",
			size = { width = 74, height = -1 },
			onClick = function()
				dlg:hide()
			end
		}
		dlg:endSizer()
	dlg:endSizer()

	local showDialogue = false -- Needed since mcn.export() seems to return true no matter what.
	
	-- add all the errors to the dialog
	local rowIndex = 1
	for i, v in errors
	do
		showDialogue = true
		listCtrl:insertItem(rowIndex, "ERROR: "..v.message)
		objects[rowIndex] = v.name
		listCtrl:setItemBold(rowIndex, true)
		rowIndex = rowIndex + 1
	end

	-- add all the warnings to the dialog
	for i, v in warnings do
		showDialogue = true
		listCtrl:insertItem(rowIndex, "WARNING: "..v.message)
		objects[rowIndex] = v.name
		rowIndex = rowIndex + 1
	end

	if(showDialogue) then
		dlg:show()
	end
end

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- INITIALIZATION
-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

-- Unregister any previously registered NM event handlers for drag and drop
EVENTS["mcOnDropText"] = {}
registerEventHandler("mcOnDropText", onDropTextAndLocation)