/* The documentation in here is platform-specific versions of function-
   -descriptions for functions in bapipew.c */

/*
 * Manipulating world object custom pipelines (platform-specific)
 *
 * Copyright (c) Criterion Software Limited
 */

/**
 * \ingroup rpworldp2sky2
 * \page rpworldp2sky2overview PS2 RpWorld Overview
 *
 * The default pipelines provided for PlayStation2 are implemented using the
 * PS2managerNode pipeline.  This is a pipeline which uses a single node on the
 * EE core and dispatches to microcode running on VU1.  To support customization
 * of this node, it exposes callback functions.  An application can use these callbacks
 * to arrange for application code to be called during the pipeline execution.
 * (This technique is therefore functionally equivalent to a series of nodes.)
 * The objallinone, bridge, and instance nodes are still available for application use;
 * RenderWare no longer uses these internally.
 *
*/

#include <rwcore.h>

#include "bapipew.h"

#include "nodeAtomicInstance.h"
#include "nodeWorldSectorInstance.h"
#include "nodeMaterialScatter.h"
#include "nodeAtomicEnumerateLights.h"
#include "nodeWorldSectorEnumerateLights.h"
#include "nodePreLight.h"
#include "nodeLight.h"
#include "nodePostLight.h"

#include "matputil.h"
#include "nodeps2manager.h"
#include "nodeps2objallinone.h"
#include "nodeps2matinstance.h"
#include "nodeps2matbridge.h"

#include "nodeps2all.h"
#include "ps2allatomic.h"
#include "ps2allsector.h"
#include "ps2allim3d.h"

#include "wrldpipe.h"


#if (!defined(DOXYGEN))
static const char rcsid[] __RWUNUSED__ =
    "@@@@(#)$Id: wrldpipe.c,v 1.120 2001/07/19 13:08:41 johns Exp $";
#endif /* (!defined(DOXYGEN)) */



/****************************************************************************
 local (static) globals
 */

RwInt32 rwPip2GeometryOffset, rwPip2AtomicOffset, rwPip2WorldSectorOffset;

/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   Functions

   !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */

/****************************************************************************
 _pluginObjectCreate()
 */

static void        *
_pluginObjectCreate(void *object,
                    RwInt32 offsetInObject,
                    RwInt32 __RWUNUSED__ sizeInObject)
{
    RWFUNCTION(RWSTRING("_pluginObjectCreate"));

    *(RwMeshCache * *) (((RwUInt8 *) object) + offsetInObject) = (RwMeshCache *)NULL;

    RWRETURN(object);
}

/****************************************************************************
 _pluginObjectDestroy()
 */

static void        *
_pluginObjectDestroy(void *object,
                     RwInt32 offsetInObject,
                     RwInt32 __RWUNUSED__ sizeInObject)
{
    RWFUNCTION(RWSTRING("_pluginObjectDestroy"));

    {
        RwMeshCache * p =
            *(RwMeshCache * *) (((RwUInt8 *) object) + offsetInObject);

        if (p != NULL)
        {
            /* Destroy the resource entries within and the array of refs */
            rpObjectDestroyMeshCache(p);
        }
    }

    RWRETURN(object);
}

/****************************************************************************
 _pluginGeometryCopy()
 */

static void *
_pluginGeometryCopy(void *dstObject,
                    const void * __RWUNUSED__ srcObject,
                    RwInt32 offsetInObject,
                    RwInt32 __RWUNUSED__ sizeInObject)
{
    const RpGeometry *srcGeom __RWUNUSEDRELEASE__=
        (const RpGeometry *)srcObject;
    RpGeometry *dstGeom = (RpGeometry *)dstObject;

    RWFUNCTION(RWSTRING("_pluginGeometryCopy"));

    if (NULL != dstGeom->mesh)
    {
        RWASSERT(NULL != srcGeom);
        RWASSERT(NULL != srcGeom->mesh);
        RWASSERT(dstGeom->mesh->numMeshes == srcGeom->mesh->numMeshes);

        if (NULL == rpGeometryGetMeshCache(dstGeom, dstGeom->mesh->numMeshes))
        {
            /* Argh, failed to created destination geometry mesh cache! */
            RWRETURN(NULL);
        }
    }
    else
    {
        *(RwMeshCache * *)(((RwUInt8 *)dstGeom) + offsetInObject) = (RwMeshCache *)NULL;
    }

    RWRETURN(dstObject);
}

/****************************************************************************
 _pluginAtomicCopy()
 */

static void        *
_pluginAtomicCopy(void *dstObject,
                  const void * __RWUNUSED__ srcObject,
                  RwInt32 offsetInObject,
                  RwInt32 __RWUNUSED__ sizeInObject)
{
    const RpAtomic *src = (const RpAtomic *)srcObject;
    RpAtomic *dst = (RpAtomic *)dstObject;
    RpGeometry *srcGeom, *dstGeom;

    RWFUNCTION(RWSTRING("_pluginAtomicCopy"));

    dstGeom = RpAtomicGetGeometry(dst);
    RWASSERT(dstGeom != NULL);

    if (NULL != dstGeom->mesh)
    {
        srcGeom = RpAtomicGetGeometry(src);
        RWASSERT(NULL != srcGeom);
        RWASSERT(NULL != srcGeom->mesh);
        RWASSERT(dstGeom->mesh->numMeshes == srcGeom->mesh->numMeshes);

        if (NULL == rpAtomicGetMeshCache(dst, dstGeom->mesh->numMeshes))
        {
            /* Argh, failed to created destination atomic mesh cache! */
            RWRETURN(NULL);
        }
    }
    else
    {
        *(RwMeshCache * *) (((RwUInt8 *) dst) + offsetInObject) = (RwMeshCache *)NULL;
    }

    RWRETURN(dstObject);
}

/****************************************************************************
 _pluginSectorCopy()
 */

static void        *
_pluginSectorCopy(void *dstObject,
                  const void *srcObject,
                  RwInt32 offsetInObject,
                  RwInt32 __RWUNUSED__ sizeInObject)
{
    const RpWorldSector *src  __RWUNUSEDRELEASE__  =
        (const RpWorldSector *)srcObject;
    RpWorldSector *dst = (RpWorldSector *)dstObject;

    RWFUNCTION(RWSTRING("_pluginSectorCopy"));

    /* WTF? We're copying a sector?? Alriiight... */

    if (NULL != dst->mesh)
    {
        RWASSERT(NULL != src->mesh);
        RWASSERT(dst->mesh->numMeshes == src->mesh->numMeshes);

        if (NULL == rpWorldSectorGetMeshCache(dst, dst->mesh->numMeshes))
        {
            /* Argh, failed to created destination sector mesh cache! */
            RWRETURN(NULL);
        }
    }
    else
    {
        *(RwMeshCache * *) (((RwUInt8 *) dst) + offsetInObject) = (RwMeshCache *)NULL;
    }

    RWRETURN(dstObject);
}

/* Copied from baworobj.c, with modifications to add mesh cache *************/

/****************************************************************************
 _initAtomicMeshCache

 Initializes an atomic's meshcache just after it's been read from a stream

 On entry   : Object (atomic)
            : Plugin data offset (not used)
            : Plugin data size (not used)
 On exit    : Size of mesh when serialised (in bytes); zero
 */

static              RwBool
_initAtomicMeshCache(void *object,
                    RwInt32 __RWUNUSED__ offsetInObject,
                    RwInt32 __RWUNUSED__ sizeInObject)
{
    RpAtomic *atomic = (RpAtomic *)object;
    RpGeometry *geom;

    RWFUNCTION(RWSTRING("_initAtomicMeshCache"));

    RWASSERT(NULL != atomic);
    geom = RpAtomicGetGeometry(atomic);
    RWASSERT(NULL != geom);

    if (NULL != geom->mesh)
    {
        if (RpGeometryGetNumMorphTargets(geom) == 1)
        {
            if (NULL == rpGeometryGetMeshCache(geom, geom->mesh->numMeshes))
            {
                RWRETURN(FALSE);
            }
        }
        else
        {
            if (NULL == rpAtomicGetMeshCache(atomic, geom->mesh->numMeshes))
            {
                RWRETURN(FALSE);
            }
        }
    }

    RWRETURN(TRUE);
}

/****************************************************************************
 _initSectorMeshCache

 Initializes a sector's meshcache just after it's been read from a stream

 On entry   : Object (sector)
            : Plugin data offset (not used)
            : Plugin data size (not used)
 On exit    : Size of mesh when serialised (in bytes); zero
 */

static              RwBool
_initSectorMeshCache(void *object,
                    RwInt32 __RWUNUSED__ offsetInObject,
                    RwInt32 __RWUNUSED__ sizeInObject)
{
    RpWorldSector *sector = (RpWorldSector *)object;

    RWFUNCTION(RWSTRING("_initSectorMeshCache"));

    if (NULL != sector->mesh)
    {
        if (NULL == rpWorldSectorGetMeshCache(sector, sector->mesh->numMeshes))
        {
            /* Argh, failed to created destination sector mesh cache! */
            RWRETURN(FALSE);
        }
    }

    RWRETURN(TRUE);
}

/****************************************************************************
 _rxWorldDevicePluginAttach()
 */

RwBool
_rxWorldDevicePluginAttach(void)
{
    RwBool              result = FALSE; /* fail, unless explicitly set TRUE */
    RwInt32             status = 0;

    RWFUNCTION(RWSTRING("_rxWorldDevicePluginAttach"));

    /* device-specific set-up... */

    rwPip2GeometryOffset = RpGeometryRegisterPlugin(sizeof(RwMeshCache *),
                                                    rwID_RXWORLDDEVICEMODULE,
                                                    _pluginObjectCreate,
                                                    _pluginObjectDestroy,
                                                    _pluginGeometryCopy);

    rwPip2AtomicOffset = RpAtomicRegisterPlugin(sizeof(RwMeshCache *),
                                                rwID_RXWORLDDEVICEMODULE,
                                                _pluginObjectCreate,
                                                _pluginObjectDestroy,
                                                _pluginAtomicCopy);

    rwPip2WorldSectorOffset =
        RpWorldSectorRegisterPlugin(sizeof(RwMeshCache *),
                                    rwID_RXWORLDDEVICEMODULE,
                                    _pluginObjectCreate,
                                    _pluginObjectDestroy,
                                    _pluginSectorCopy);

    /* We make sure that meshcaches are set up (this does a malloc and
     * we're trying to avoid fragmentation caused by allocation when
     * objects are first rendered) for atomics and sectors
     *  o when they are read from a stream (below)
     *  o when they are copied (above)
     * ...both of which likely happen at app startup so fragmentation
     * is minimised. We also need to have normal constructors/destructors
     * for the meshcaches. */
    status |= RpAtomicSetStreamAlwaysCallBack(
                  rwID_RXWORLDDEVICEMODULE, _initAtomicMeshCache);
    status |= RpWorldSectorSetStreamAlwaysCallBack(
                  rwID_RXWORLDDEVICEMODULE, _initSectorMeshCache);

    if (!((rwPip2GeometryOffset | rwPip2AtomicOffset |
           rwPip2WorldSectorOffset | status) < 0))
    {
        result = TRUE;
    }

    RWRETURN(result);
}

/* The four flavours of WorldSector instance pipe
 *  - PS2All, PS2Manager, AllInOne and vanilla */

static RxPipeline *
CreatePS2AllWorldSectorPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreatePS2AllWorldSectorPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition   *ps2all = RxNodeDefinitionGetPS2All();
            RxPipelineNode     *plnode;

            lpipe = RxLockedPipeAddFragment(lpipe,
                                            (RwUInt32 *)NULL,
                                            ps2all,
                                            (RxNodeDefinition *)NULL);

            RWASSERT(lpipe != NULL);

            lpipe = RxLockedPipeUnlock(lpipe);

            if (lpipe != NULL)
            {
                plnode = RxPipelineFindNodeByName(lpipe,
                                                  ps2all->name, 
                                                  (RxPipelineNode *)NULL, 
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode!=NULL);

                /* Set up the necessary callbacks */
                RxPipelineNodePS2AllSetCallBack(
                    plnode, rxPS2ALLCALLBACKOBJECTSETUP,
                    RpWorldSectorPS2AllObjectSetupCallBack);

                RWRETURN(pipe);
            }
        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "PS2AllWorldSector"));

    RWRETURN((RxPipeline *)NULL);
}

static RxPipeline *
CreatePS2ManagerWorldSectorPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreatePS2ManagerWorldSectorPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition   *manager =
                RxNodeDefinitionGetPS2Manager(rxOBJTYPE_WORLDSECTOR);
            RxPipelineNode     *plnode, *result;

            lpipe = RxLockedPipeAddFragment(lpipe,
                                            (RwUInt32 *)NULL,
                                            manager,
                                            (RxNodeDefinition *)NULL);

            RWASSERT(lpipe != NULL);

            plnode = RxPipelineFindNodeByName(lpipe,
                                              manager->name,
                                              (RxPipelineNode *)NULL,
                                              (RwInt32 *)NULL);
            RWASSERT(plnode != NULL);
            /* get manager node to generate cluster "xyz" */
            result = RxPipelineNodePS2ManagerGenerateCluster(
                         plnode, &RxClPS2xyz, CL_XYZ);
            RWASSERT(result != NULL);

            /* get manager node to generate cluster "uv" */
            result = RxPipelineNodePS2ManagerGenerateCluster(
                         plnode, &RxClPS2uv, CL_UV);
            RWASSERT(result != NULL);

            /* get manager node to generate cluster "rgba" */
            result = RxPipelineNodePS2ManagerGenerateCluster(
                         plnode, &RxClPS2rgba, CL_RGBA);
            RWASSERT(result != NULL);

            /* get manager node to generate cluster "normal" */
            result = RxPipelineNodePS2ManagerGenerateCluster(
                          plnode, &RxClPS2normal, CL_NORMAL);
            RWASSERT(result != NULL);

            RxPipelineNodePS2ManagerSetVUBufferSizes(plnode,
                                              _rwskyStrideOfInputCluster,
                                              _rwskyTSVertexCount,
                                              _rwskyTLTriCount);

            lpipe = RxLockedPipeUnlock(lpipe);

            if (lpipe != NULL)
            {
                plnode = RxPipelineFindNodeByName(lpipe,
                                                  manager->name,
                                                  (RxPipelineNode *)NULL,
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode!=NULL);

                RxPipelineNodePS2ManagerSetVIFOffset(plnode,
                                                     _rwskyVIFOffset);

                /* Hack to change VU1 code array index calculation - due
                 * to our new default VU1 code array being smaller since
                 * putting in backface culling support. Deliberately no
                 * wrapper function for this. All other pipes should
                 * work exactly as before. */
                ((RxPipelineNodePS2ObjAllInOneData *)plnode->privateData)
                    ->genericVU1Index = TRUE;

                RWRETURN(pipe);
            }
        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "PS2ManagerWorldSector"));

    RWRETURN((RxPipeline *)NULL);
}

static RxPipeline *
CreateAllInOneWorldSectorPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreateAllInOneWorldSectorPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition   *objallinone =
                RxNodeDefinitionGetPS2ObjAllInOne(rxOBJTYPE_WORLDSECTOR);
            RxNodeDefinition   *instance =
                RxNodeDefinitionGetPS2MatInstance();
            RxNodeDefinition   *bridge =
                RxNodeDefinitionGetPS2MatBridge();
            RxPipelineNode     *plnode;
            RxPipelineNode     *result;

            lpipe = RxLockedPipeAddFragment(lpipe,
                                            (RwUInt32 *)NULL,
                                            objallinone,
                                            instance,
                                            bridge,
                                            (RxNodeDefinition *)NULL);

            RWASSERT(lpipe != NULL);

            /* get instance node to generate cluster "xyz" */
            plnode = RxPipelineFindNodeByName(lpipe,
                                              instance->name,
                                              (RxPipelineNode *)NULL,
                                              (RwInt32 *)NULL);
            RWASSERT(plnode != NULL);
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2xyz, CL_XYZ);
            RWASSERT(result != NULL);

            /* get instance node to generate cluster "uv" */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2uv, CL_UV);
            RWASSERT(result != NULL);

            /* get instance node to generate cluster "rgba" */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2rgba, CL_RGBA);
            RWASSERT(result != NULL);

            /* get instance node to generate cluster "normal" */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2normal, CL_NORMAL);
            RWASSERT(result != NULL);

            RxPipelineNodePS2MatInstanceNodeSetVUBufferSizes(plnode,
                                         _rwskyStrideOfInputCluster,
                                         _rwskyTSVertexCount,
                                         _rwskyTLTriCount);

            lpipe = RxLockedPipeUnlock(lpipe);

            if (lpipe != NULL)
            {

                plnode = RxPipelineFindNodeByName(lpipe,
                                                  bridge->name,
                                                  (RxPipelineNode *)NULL,
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode!=NULL);

                RxBridgeNodeSetVIFOffset(plnode, _rwskyVIFOffset);

                /* Get the instance node to dispatch per-mesh packets to the
                 * pipelines specified by their associated materials */
                plnode = RxPipelineFindNodeByName(lpipe,
                                                  objallinone->name,
                                                  (RxPipelineNode *)NULL,
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode != NULL);
                result =
                    RxPipelineNodePS2ObjAllInOneSetGrouping(plnode, TRUE);
                RWASSERT(result != NULL);

                RWRETURN(pipe);
            }
        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "AllInOneWorldSector"));

    RWRETURN((RxPipeline *)NULL);
}

static RxPipeline *
CreateVanillaWorldSectorPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreateVanillaWorldSectorPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition *objallinone =
                RxNodeDefinitionGetPS2ObjAllInOne(rxOBJTYPE_WORLDSECTOR);

            lpipe = RxLockedPipeAddFragment(lpipe,
                                            (RwUInt32 *)NULL,
                                            objallinone,
                                            (RxNodeDefinition *)NULL);

            RWASSERT(lpipe != NULL);

            lpipe = RxLockedPipeUnlock(lpipe);

            if (lpipe != NULL)
            {
                RWRETURN(pipe);
            }
        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "VanillaWorldSector"));

    RWRETURN((RxPipeline *)NULL);
}

/**
 * \ingroup rpworldsectorsky2
 * \ref RpWorldSectorSkyGetPS2AllPipeline returns a
 * pointer to the default world-sector PS2All pipeline.
 *
 * The world-sector pipeline based on the PS2All.csl node renders
 * world-sectors using just this node, in a similar fashion to
 * PS2Manager.csl. The differences are that PS2All.csl does not
 * ignore material pipelines (with the restriction that they
 * must be constructed from the PS2AllMat.csl node; see
 * \ref RxNodeDefinitionGetPS2All and \ref RxNodeDefinitionGetPS2AllMat
 * for details) and PS2All.csl is far more customisable with
 * more available callbacks than PS2Manager.csl. The aim here is
 * not so much to optimize the default RW RpAtomic and RpWorldSector
 * pipelines but rather to allow developers to optimize pipelines
 * based on the specifics of objects and vector code used in their
 * game.
 *
 * If you wish to submit 2D triangles through Submit.csl (i.e you want to
 * do transformation and lighting and whatever else CPU-side - which will
 * of course be very slow) then rather than use this world sector instance
 * pipeline, you should use a generic (non-PS2-specific) one.
 *
 * The world plugin must be attached before using this function.
 *
 * The PS2All world-sector instance pipeline:
 *
 *   \li PS2All.csl
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2All
 * \see RxNodeDefinitionGetPS2AllMat
 * \see RxNodeDefinitionGetPS2Manager
 * \see RxPipelineNodePS2ManagerGenerateCluster
 * \see RxPipelineNodePS2ManagerSetVUBufferSizes
 * \see RxPipelineNodePS2ManagerSetPointListVUBufferSize
 * \see RxPipelineNodePS2ManagerSetVIFOffset
 * \see RxPipelineNodePS2ManagerSetVU1CodeArray
 * \see RxPipelineNodePS2ManagerSetLighting
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 */
RxPipeline *
RpWorldSectorSkyGetPS2AllPipeline(void)
{
    RWAPIFUNCTION(RWSTRING("RpWorldSectorSkyGetPS2AllPipeline"));

    RWRETURN(RXPIPELINEGLOBAL(ps2AllWorldSectorPipeline));
}

/**
 * \ingroup rpworldsectorsky2
 * \ref RpWorldSectorSkyGetPS2ManagerPipeline returns a
 * pointer to the default world-sector PS2Manager pipeline.
 *
 * The world-sector pipeline based on the PS2Manager node renders
 * world-sectors using just this node (shrinking the pipe to a single
 * node is a significant speed win on PS2), ignoring material pipelines,
 * so all meshes in a sector will be rendered in the same style (that
 * defined by the sector pipeline). This node will be made much more
 * customisable in future releases such that it will be all you need
 * on PS2 (bar nodes like PVSWorldSector.csl).
 *
 * If you wish to submit 2D triangles through Submit.csl (i.e you want to
 * do transformation and lighting and whatever else CPU-side - which will
 * of course be very slow) then rather than use this world sector instance
 * pipeline, you should use a generic (non-PS2-specific) one.
 *
 * The world plugin must be attached before using this function.
 *
 * The PS2Manager world-sector instance pipeline:
 *
 *   \li PS2Manager.csl
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2Manager
 * \see RxPipelineNodePS2ManagerGenerateCluster
 * \see RxPipelineNodePS2ManagerSetVUBufferSizes
 * \see RxPipelineNodePS2ManagerSetPointListVUBufferSize
 * \see RxPipelineNodePS2ManagerSetVIFOffset
 * \see RxPipelineNodePS2ManagerSetVU1CodeArray
 * \see RxPipelineNodePS2ManagerSetLighting
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 */
RxPipeline *
RpWorldSectorSkyGetPS2ManagerPipeline(void)
{
    RWAPIFUNCTION(RWSTRING("RpWorldSectorSkyGetPS2ManagerPipeline"));

    RWRETURN(RXPIPELINEGLOBAL(platformWorldSectorPipeline));
}

/**
 * \ingroup rpworldsectorsky2
 * \ref RpWorldSectorSkyGetAllInOnePipeline
 * returns a pointer to the default world-sector AllInOne pipeline.
 *
 * This world-sector instance pipeline is effectively the vanilla instance
 * pipeline concatenated with the default material render pipeline. This
 * pipeline is somewhat less flexible than the split instance/material
 * pipelines, since the whole world-sector is rendered the same way -
 * material pipelines are ignored. However, it executes faster because
 * passing packets between pipelines is quite slow. Any rendering effect
 * can still be achieved (uniformly across the whole object) by simply
 * concatenating any custom material pipeline onto the PS2ObjAllInOne.csl
 * node and using the \ref RxPipelineNodePS2ObjAllInOneSetGrouping node API
 * function to tell this node to send all meshes in the object down the rest
 * of the instance pipeline rather than to pipelines specified by the
 * materials of the object.
 *
 * If you wish to submit 2D triangles through Submit.csl (i.e you want to
 * do transformation and lighting and whatever else CPU-side - which will
 * of course be very slow) then rather than use this world sector instance
 * pipeline, you should use a generic (non-PS2-specific) one.
 *
 * The world plugin must be attached before using this function.
 *
 * The PS2 all-in-one world-sector instance pipeline:
 * \verbatim
     PS2ObjAllInOne.csl
      v
     PS2MatInstance.csl
      v
     PS2MatBridge.csl
   \endverbatim
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2ObjAllInOne
 * \see RxPipelineNodePS2ObjAllInOneSetGrouping
 * \see RxNodeDefinitionGetPS2MatInstance
 * \see RxNodeDefinitionGetPS2MatBridge
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 */
RxPipeline *
RpWorldSectorSkyGetAllInOnePipeline(void)
{
    RxPipeline *result;
    RWAPIFUNCTION(RWSTRING("RpWorldSectorSkyGetAllInOnePipeline"));

    result =  RXPIPELINEGLOBAL(allInOneWorldSectorPipeline);

    RWRETURN(result);
}

/**
 * \ingroup rpworldsectorsky2
 * \ref RpWorldSectorSkyGetVanillaPipeline
 * returns a pointer to the default world-sector vanilla pipeline.
 *
 * This world-sector instance pipeline is the 'vanilla' one (i.e no
 * tricks have been used to trade off flexibility and speed - see
 * \ref RpWorldSectorSkyGetPS2ManagerPipeline and
 * \ref RpWorldSectorSkyGetAllInOnePipeline) and it deals with per-object
 * data (transformation matrix, lights, etc) before passing one packet
 * per mesh to material-specified render pipelines.
 *
 * If you wish to submit 2D triangles through Submit.csl (i.e you want to
 * do transformation and lighting and whatever else CPU-side - which will
 * of course be very slow) then rather than use this world sector instance
 * pipeline, you should use a generic (non-PS2-specific) one.
 *
 * The world plugin must be attached before using this function.
 *
 * \verbatim
   The PS2 vanilla world-sector instance pipeline:

     PS2ObjAllInOne.csl
   \endverbatim
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2ObjAllInOne
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 */
RxPipeline *
RpWorldSectorSkyGetVanillaPipeline(void)
{
    RWAPIFUNCTION(RWSTRING("RpWorldSectorSkyGetVanillaPipeline"));

    RWRETURN(RXPIPELINEGLOBAL(vanillaWorldSectorPipeline));
}

/* The four flavours of Atomic instance pipe
 *  - PS2All, PS2Manager, AllInOne and vanilla */

static RxPipeline *
CreatePS2AllAtomicPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreatePS2AllAtomicPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition   *ps2all = RxNodeDefinitionGetPS2All();
            RxPipelineNode     *plnode;

            lpipe = RxLockedPipeAddFragment(lpipe,
                                            (RwUInt32 *)NULL,
                                            ps2all,
                                            (RxNodeDefinition *)NULL);

            RWASSERT(lpipe != NULL);

            lpipe = RxLockedPipeUnlock(lpipe);

            if (lpipe != NULL)
            {
                plnode = RxPipelineFindNodeByName(lpipe,
                                                  ps2all->name, 
                                                  (RxPipelineNode *)NULL, 
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode!=NULL);

                /* Set up the necessary callbacks */
                RxPipelineNodePS2AllSetCallBack(
                    plnode, rxPS2ALLCALLBACKOBJECTSETUP,
                    RpAtomicPS2AllObjectSetupCallBack);
                RxPipelineNodePS2AllSetCallBack(
                    plnode, rxPS2ALLCALLBACKOBJECTFINALIZE,
                    RpAtomicPS2AllObjectFinalizeCallBack);

                RWRETURN(pipe);
            }
        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "PS2AllAtomic"));

    RWRETURN((RxPipeline *)NULL);
}

static RxPipeline         *
CreatePS2ManagerAtomicPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreatePS2ManagerAtomicPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition *manager =
                RxNodeDefinitionGetPS2Manager(rxOBJTYPE_ATOMIC);
            RxPipelineNode   *plnode, *result;

            lpipe = RxLockedPipeAddFragment(lpipe,
                                            (RwUInt32 *)NULL,
                                            manager,
                                            (RxNodeDefinition *)NULL);

            RWASSERT(lpipe != NULL);

            plnode = RxPipelineFindNodeByName(lpipe,
                                              manager->name,
                                              (RxPipelineNode *)NULL,
                                              (RwInt32 *)NULL);
            RWASSERT(plnode != NULL);
            /* get manager node to generate cluster "xyz" */
            result = RxPipelineNodePS2ManagerGenerateCluster(
                         plnode, &RxClPS2xyz, CL_XYZ);
            RWASSERT(result != NULL);

            /* get manager node to generate cluster "uv" */
            result = RxPipelineNodePS2ManagerGenerateCluster(
                         plnode, &RxClPS2uv, CL_UV);
            RWASSERT(result != NULL);

            /* get manager node to generate cluster "rgba" */
            result = RxPipelineNodePS2ManagerGenerateCluster(
                         plnode, &RxClPS2rgba, CL_RGBA);
            RWASSERT(result != NULL);

            /* get manager node to generate cluster "normal" */
            result = RxPipelineNodePS2ManagerGenerateCluster(
                         plnode, &RxClPS2normal, CL_NORMAL);
            RWASSERT(result != NULL);

            RxPipelineNodePS2ManagerSetVUBufferSizes(plnode,
                                         _rwskyStrideOfInputCluster,
                                         _rwskyTSVertexCount,
                                         _rwskyTLTriCount);

            lpipe = RxLockedPipeUnlock(lpipe);

            if (lpipe != NULL)
            {
                plnode = RxPipelineFindNodeByName(lpipe,
                                                  manager->name,
                                                  (RxPipelineNode *)NULL,
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode!=NULL);

                RxPipelineNodePS2ManagerSetVIFOffset(plnode,
                                                     _rwskyVIFOffset);

                /* Hack to change VU1 code array index calculation - due
                 * to our new default VU1 code array being smaller since
                 * putting in backface culling support. Deliberately no
                 * wrapper function for this. All other pipes should
                 * work exactly as before. */
                ((RxPipelineNodePS2ObjAllInOneData *)plnode->privateData)
                    ->genericVU1Index = TRUE;

                RWRETURN(pipe);
             }
        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "PS2ManagerAtomic"));

    RWRETURN((RxPipeline *)NULL);
}

static RxPipeline         *
CreateAllInOneAtomicPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreateAllInOneAtomicPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition   *objallinone =
                RxNodeDefinitionGetPS2ObjAllInOne(rxOBJTYPE_ATOMIC);
            RxNodeDefinition   *instance =
                RxNodeDefinitionGetPS2MatInstance();
            RxNodeDefinition   *bridge =
                RxNodeDefinitionGetPS2MatBridge();
            RxPipelineNode     *plnode;
            RxPipelineNode     *result;

            lpipe = RxLockedPipeAddFragment(lpipe,
                                            (RwUInt32 *)NULL,
                                            objallinone,
                                            instance,
                                            bridge,
                                            (RxNodeDefinition *)NULL);

            RWASSERT(lpipe != NULL);

            /* get instance node to generate cluster "xyz" */
            plnode = RxPipelineFindNodeByName(lpipe,
                                              instance->name,
                                              (RxPipelineNode *)NULL,
                                              (RwInt32 *)NULL);
            RWASSERT(plnode != NULL);
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2xyz, CL_XYZ);
            RWASSERT(result != NULL);

            /* get instance node to generate cluster "uv" */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2uv, CL_UV);
            RWASSERT(result != NULL);

            /* get instance node to generate cluster "rgba" */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2rgba, CL_RGBA);
            RWASSERT(result != NULL);

            /* get instance node to generate cluster "normal" */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2normal, CL_NORMAL);
            RWASSERT(result != NULL);

            RxPipelineNodePS2MatInstanceNodeSetVUBufferSizes(plnode,
                                         _rwskyStrideOfInputCluster,
                                         _rwskyTSVertexCount,
                                         _rwskyTLTriCount);

            lpipe = RxLockedPipeUnlock(lpipe);

            if (lpipe != NULL)
            {
                plnode = RxPipelineFindNodeByName(lpipe,
                                                  bridge->name,
                                                  (RxPipelineNode *)NULL,
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode!=NULL);

                RxBridgeNodeSetVIFOffset(plnode, _rwskyVIFOffset);

                /* Get the instance node to dispatch per-mesh packets to the
                 * pipelines specified by their associated materials */
                plnode = RxPipelineFindNodeByName(lpipe,
                                                  objallinone->name,
                                                  (RxPipelineNode *)NULL,
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode != NULL);
                result =
                    RxPipelineNodePS2ObjAllInOneSetGrouping(plnode, TRUE);
                RWASSERT(result != NULL);

                RWRETURN(pipe);
            }

        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "AllInOneAtomic"));

    RWRETURN((RxPipeline *)NULL);
}

static RxPipeline         *
CreateVanillaAtomicPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreateVanillaAtomicPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition *objallinone =
                RxNodeDefinitionGetPS2ObjAllInOne(rxOBJTYPE_ATOMIC);

            lpipe = RxLockedPipeAddFragment(lpipe,
                                             (RwUInt32 *)NULL,
                                            objallinone,
                                            (RxNodeDefinition *)NULL);

            RWASSERT(lpipe != NULL);

            lpipe = RxLockedPipeUnlock(lpipe);

            if (lpipe != NULL)
            {
                RWRETURN(pipe);
            }
        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "VanillaAtomic"));

    RWRETURN((RxPipeline *)NULL);
}

/**
 * \ingroup rpatomicsky2
 * \ref RpAtomicSkyGetPS2AllPipeline
 * returns a pointer to the default atomic PS2All pipeline.
 *
 * The atomic pipeline based on the PS2All.csl node renders
 * atomics using just this node, in a similar fashion to
 * PS2Manager.csl. The differences are that PS2All.csl does not
 * ignore material pipelines (with the restriction that they
 * must be constructed from the PS2AllMat.csl node; see
 * \ref RxNodeDefinitionGetPS2All and \ref RxNodeDefinitionGetPS2AllMat
 * for details) and PS2All.csl is far more customisable with
 * more available callbacks than PS2Manager.csl. The aim here is
 * not so much to optimize the default RW RpAtomic and RpWorldSector
 * pipelines but rather to allow developers to optimize pipelines
 * based on the specifics of objects and vector code used in their
 * game.
 *
 * If you wish to submit 2D triangles through Submit.csl (i.e you want to
 * do transformation and lighting and whatever else CPU-side - which will
 * of course be very slow) then rather than use this atomic instance
 * pipeline, you should use a generic (non-PS2-specific) one.
 *
 * The world plugin must be attached before using this function.
 *
 * The PS2Atomic atomic instance pipeline:
 *
 *   PS2All.csl
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2All
 * \see RxNodeDefinitionGetPS2AllMat
 * \see RxNodeDefinitionGetPS2Manager
 * \see RxPipelineNodePS2ManagerGenerateCluster
 * \see RxPipelineNodePS2ManagerSetVUBufferSizes
 * \see RxPipelineNodePS2ManagerSetPointListVUBufferSize
 * \see RxPipelineNodePS2ManagerSetVIFOffset
 * \see RxPipelineNodePS2ManagerSetVU1CodeArray
 * \see RxPipelineNodePS2ManagerSetLighting
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 */
RxPipeline *
RpAtomicSkyGetPS2AllPipeline(void)
{
    RWAPIFUNCTION(RWSTRING("RpAtomicSkyGetPS2AllPipeline"));

    RWRETURN(RXPIPELINEGLOBAL(ps2AllAtomicPipeline));
}

/**
 * \ingroup rpatomicsky2
 * \ref RpAtomicSkyGetPS2ManagerPipeline
 * returns a pointer to the default atomic PS2Manager pipeline.
 *
 * The atomic pipeline based on the PS2Manager node renders
 * atomics using just this node (shrinking the pipe to a single
 * node is a significant speed win on PS2), ignoring material pipelines,
 * so all meshes in an atomic will be rendered in the same style (that
 * defined by the atomic pipeline). This node will be made much more
 * customisable in future releases such that it will be all you need
 * on PS2 (bar nodes like PVSWorldSector.csl).
 *
 * If you wish to submit 2D triangles through Submit.csl (i.e you want to
 * do transformation and lighting and whatever else CPU-side - which will
 * of course be very slow) then rather than use this atomic instance
 * pipeline, you should use a generic (non-PS2-specific) one.
 *
 * The world plugin must be attached before using this function.
 *
 * The PS2Manager atomic instance pipeline:
 *
 *   PS2Manager.csl
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2Manager
 * \see RxPipelineNodePS2ManagerGenerateCluster
 * \see RxPipelineNodePS2ManagerSetVUBufferSizes
 * \see RxPipelineNodePS2ManagerSetPointListVUBufferSize
 * \see RxPipelineNodePS2ManagerSetVIFOffset
 * \see RxPipelineNodePS2ManagerSetVU1CodeArray
 * \see RxPipelineNodePS2ManagerSetLighting
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 */
RxPipeline *
RpAtomicSkyGetPS2ManagerPipeline(void)
{
    RWAPIFUNCTION(RWSTRING("RpAtomicSkyGetPS2ManagerPipeline"));

    RWRETURN(RXPIPELINEGLOBAL(platformAtomicPipeline));
}

/**
 * \ingroup rpatomicsky2
 * \ref RpAtomicSkyGetAllInOnePipeline
 * returns a pointer to the default atomic AllInOne pipeline.
 *
 * This atomic instance pipeline is effectively the vanilla instance
 * pipeline concatenated with the default material render pipeline. This
 * pipeline is somewhat less flexible than the split instance/material
 * pipelines, since the whole atomic is rendered the same way - material
 * pipelines are ignored. However, it executes faster because passing
 * packets between pipelines is quite slow. Any rendering effect can still
 * be achieved (uniformly across the whole object) by simply concatenating
 * any custom material pipeline onto the PS2ObjAllInOne.csl node and using
 * the \ref RxPipelineNodePS2ObjAllInOneSetGrouping node API function to
 * tell this node to send all meshes in the object down the rest of the
 * instance pipeline rather than to pipelines specified by the materials of
 * the object.
 *
 * If you wish to submit 2D triangles through Submit.csl (i.e you want to
 * do transformation and lighting and whatever else CPU-side - which will
 * of course be very slow) then rather than use this atomic instance
 * pipeline, you should use a generic (non-PS2-specific) one.
 *
 * The world plugin must be attached before using this function.
 *
 * The PS2 all-in-one atomic instance pipeline:
 * \verbatim
     PS2ObjAllInOne.csl
      v
     PS2MatInstance.csl
      v
     PS2MatBridge.csl
   \endverbatim
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2ObjAllInOne
 * \see RxNodeDefinitionGetPS2MatInstance
 * \see RxNodeDefinitionGetPS2MatBridge
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 */
RxPipeline *
RpAtomicSkyGetAllInOnePipeline(void)
{
    RWAPIFUNCTION(RWSTRING("RpAtomicSkyGetAllInOnePipeline"));

    RWRETURN(RXPIPELINEGLOBAL(allInOneAtomicPipeline));
}

/**
 * \ingroup rpatomicsky2
 * \ref RpAtomicSkyGetVanillaPipeline
 * returns a pointer to the default atomic Vanilla pipeline.
 *
 * This atomic instance pipeline is the 'vanilla' one (i.e no
 * tricks have been used to trade off flexibility and speed - see
 * \ref RpWorldSectorSkyGetPS2ManagerPipeline and
 * \ref RpWorldSectorSkyGetAllInOnePipeline) and it deals with per-object
 * data (transformation matrix, lights, etc) before passing one packet
 * per mesh to material-specified render pipelines.
 *
 * If you wish to submit 2D triangles through Submit.csl (i.e you want to
 * do transformation and lighting and whatever else CPU-side - which will
 * of course be very slow) then rather than use this atomic instance
 * pipeline, you should use a generic (non-PS2-specific) one.
 *
 * The world plugin must be attached before using this function.
 *
 * The PS2 vanilla atomic instance pipeline:
 *
 *  PS2ObjAllInOne.csl
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2ObjAllInOne
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 */
RxPipeline *
RpAtomicSkyGetVanillaPipeline(void)
{
    RWAPIFUNCTION(RWSTRING("RpAtomicSkyGetVanillaPipeline"));

    RWRETURN(RXPIPELINEGLOBAL(vanillaAtomicPipeline));
}


/* The two flavours of material render pipe
 *  - PS2AllMat, PS2MatInstance->PS2MatBridge */
extern long vu1nullTrans   __attribute__((section(".vudata")));
static RxPipeline *
CreatePS2AllMatMaterialPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreatePS2AllMatMaterialPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition   *ps2allmat = RxNodeDefinitionGetPS2AllMat();
            RxPipelineNode     *plnode, *result;

            lpipe = RxLockedPipeAddFragment(lpipe,
                                            (RwUInt32 *)NULL,
                                            ps2allmat,
                                            (RxNodeDefinition *)NULL);

            RWASSERT(lpipe != NULL);

            plnode = RxPipelineFindNodeByName(lpipe, 
                                              ps2allmat->name, 
                                              (RxPipelineNode *)NULL, 
                                              (RwInt32 *)NULL);
            RWASSERT(plnode != NULL);

            /* get manager node to generate cluster "xyz" */
            result = RxPipelineNodePS2AllMatGenerateCluster(
                         plnode, &RxClPS2xyz, CL_XYZ);
            RWASSERT(result != NULL);

            /* get manager node to generate cluster "uv" */
            result = RxPipelineNodePS2AllMatGenerateCluster(
                         plnode, &RxClPS2uv, CL_UV);
            RWASSERT(result != NULL);

            /* get manager node to generate cluster "rgba" */
            result = RxPipelineNodePS2AllMatGenerateCluster(
                         plnode, &RxClPS2rgba, CL_RGBA);
            RWASSERT(result != NULL);

            /* get manager node to generate cluster "normal" */
            result = RxPipelineNodePS2AllMatGenerateCluster(
                          plnode, &RxClPS2normal, CL_NORMAL);
            RWASSERT(result != NULL);


            /* set ps2all buffer sizes */
            result = RxPipelineNodePS2AllMatSetVUBufferSizes(plnode,
                                              _rwskyStrideOfInputCluster,
                                              _rwskyTSVertexCount,
                                              _rwskyTLTriCount);
            RWASSERT(result != NULL);

            lpipe = RxLockedPipeUnlock(lpipe);

            if (lpipe != NULL)
            {
                plnode = RxPipelineFindNodeByName(lpipe,
                                                  ps2allmat->name, 
                                                  (RxPipelineNode *)NULL, 
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode!=NULL);

                plnode = RxPipelineNodePS2AllMatSetVIFOffset(plnode,
                                                 _rwskyVIFOffset);
                RWASSERT(NULL != plnode);

                /* Default VU1 code array is OK */

                /* Set up the necessary callbacks */
                RxPipelineNodePS2AllMatSetCallBack(
                    plnode, rxPS2ALLMATCALLBACKMESHINSTANCETEST,
                    RpMeshPS2AllMeshInstanceTestCallBack);
                RxPipelineNodePS2AllMatSetCallBack(
                    plnode, rxPS2ALLMATCALLBACKRESENTRYALLOC,
                    RpMeshPS2AllResEntryAllocCallBack);
                RxPipelineNodePS2AllMatSetCallBack(
                    plnode, rxPS2ALLMATCALLBACKINSTANCE,
                    RpMeshPS2AllInstanceCallBack);
                RxPipelineNodePS2AllMatSetCallBack(
                    plnode, rxPS2ALLMATCALLBACKBRIDGE,
                    RpMeshPS2AllBridgeCallBack);
#if (defined(RWMETRICS))
                RxPipelineNodePS2AllMatSetCallBack(
                    plnode, rxPS2ALLMATCALLBACKPOSTMESH,
                    RpMeshPS2AllPostMeshCallBack);
#endif /* (defined(RWMETRICS)) */

                RWRETURN(pipe);
            }
        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "PS2AllMatMaterial"));

    RWRETURN((RxPipeline *)NULL);
}

static RxPipeline *
CreateVanillaMaterialPipeline(void)
{
    RxPipeline         *pipe;

    RWFUNCTION(RWSTRING("CreateVanillaMaterialPipeline"));

    pipe = RxPipelineCreate();
    if (pipe)
    {
        RxLockedPipe       *lpipe;

        lpipe = RxPipelineLock(pipe);

        if (NULL != lpipe)
        {
            RxNodeDefinition   *instance =
                RxNodeDefinitionGetPS2MatInstance();
            RxNodeDefinition   *bridge =
                RxNodeDefinitionGetPS2MatBridge();
            RxPipelineNode     *plnode;
            RxPipelineNode     *result;

            lpipe = RxLockedPipeAddFragment(lpipe,
                                            (RwUInt32 *)NULL,
                                            instance,
                                            bridge,
                                            (RxNodeDefinition *)NULL);

            /* get instance node to generate cluster "xyz" */
            plnode = RxPipelineFindNodeByName(lpipe,
                                              instance->name,
                                              (RxPipelineNode *)NULL,
                                              (RwInt32 *)NULL);
            RWASSERT(plnode != NULL);
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2xyz, CL_XYZ);
            RWASSERT(result != NULL);

            /* get instance node to generate cluster "uv" */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2uv, CL_UV);
            RWASSERT(result != NULL);

            /* get instance node to generate cluster "rgba" */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2rgba, CL_RGBA);
            RWASSERT(result != NULL);

            /* get instance node to generate cluster "normal" */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &RxClPS2normal, CL_NORMAL);
            RWASSERT(result != NULL);

#if 0
            /* Example for adding user clusters; get instance
             * node to generate cluster "user1": */
            result = RxPipelineNodePS2MatInstanceGenerateCluster(
                         plnode, &clusterPS2user1, CL_USER1);
            RWASSERT(result != NULL);
#endif

            RxPipelineNodePS2MatInstanceNodeSetVUBufferSizes(plnode,
                                         _rwskyStrideOfInputCluster,
                                         _rwskyTSVertexCount,
                                         _rwskyTLTriCount);

            lpipe = RxLockedPipeUnlock(lpipe);

            RWASSERT(pipe == (RxPipeline *)lpipe);

            if (lpipe != NULL)
            {
                plnode = RxPipelineFindNodeByName(lpipe,
                                                  bridge->name,
                                                  (RxPipelineNode *)NULL,
                                                  (RwInt32 *)NULL);
                RWASSERT(plnode!=NULL);

                RxBridgeNodeSetVIFOffset(plnode,_rwskyVIFOffset);


                RWRETURN(pipe);
            }
        }

        RxPipelineDestroy(pipe);
    }

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "VanillaMaterial"));

    RWRETURN((RxPipeline *)NULL);
}

/**
 * \ingroup rpmaterialsky2
 * \ref RpMaterialSkyGetPS2AllMatPipeline
 * returns a pointer to the standard material PS2AllMat pipeline.
 *
 * PS2AllMat.csl is the material render pipeline node counterpart
 * to (the object instance pipeline node) PS2All.csl. Objects for
 * which this material pipeline is used must have their instance
 * pipeline constructed from PS2All.csl (see
 * \ref RxNodeDefinitionGetPS2All and \ref RxNodeDefinitionGetPS2AllMat
 * for details).
 *
 * Note that you may write your own VU1 code and attach it to a material
 * pipeline so that you can perform whatever special effects you like on
 * VU1 (see the PS2AllMat.csl API for details).
 *
 * If you wish to submit 2D triangles through Submit.csl (i.e you want to
 * do transformation and lighting and whatever else CPU-side - which will
 * of course be very slow) then rather than use this material render
 * pipeline, you should use a generic (non-PS2-specific) one.
 *
 * The world plugin must be attached before using this function.
 *
 * The PS2AllMat material render pipeline:
 *
 *   PS2Mat.csl
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2AllMat
 * \see RxNodeDefinitionGetPS2All
 * \see RxNodeDefinitionGetPS2Manager
 * \see RpMaterialGetDefaultRenderPipeline
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 */
RxPipeline *
RpMaterialSkyGetPS2AllMatPipeline(void)
{
    RWAPIFUNCTION(RWSTRING("RpMaterialSkyGetPS2AllMatPipeline"));

    RWRETURN(RXPIPELINEGLOBAL(ps2AllMatMaterialPipeline));
}

/**
 * \ingroup rpmaterialsky2
 * \ref RpMaterialSkyGetDefaultPS2AllMatPipeline
 * returns a pointer to the current default material PS2AllMat pipeline.
 *
 * This pipeline is used by default by PS2All.csl for RpMaterials which
 * have NULL RxPipeline pointers. See \ref RpMaterialSkyGetPS2AllMatPipeline
 * for further details on PS2AllMat.csl.
 *
 * The world plugin must be attached before using this function.
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RxNodeDefinitionGetPS2AllMat
 * \see RxNodeDefinitionGetPS2All
 * \see RxNodeDefinitionGetPS2Manager
 * \see RpMaterialGetDefaultRenderPipeline
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 */
RxPipeline *
RpMaterialSkyGetDefaultPS2AllMatPipeline(void)
{
    RWAPIFUNCTION(RWSTRING("RpMaterialSkyGetDefaultPS2AllMatPipeline"));

    RWRETURN(RXPIPELINEGLOBAL(currentPS2AllMatMaterialPipeline));
}

/**
 * \ingroup rpmaterialsky2
 * \ref RpMaterialSkySetDefaultPS2AllMatPipeline
 * sets the current default material PS2AllMat pipeline.
 *
 * This function defines which pipeline is to be used by default by
 * PS2All.csl for RpMaterials which have NULL RxPipeline pointers.
 * See \ref RpMaterialSkyGetPS2AllMatPipeline for further details on
 * PS2AllMat.csl.
 *
 * The world plugin must be attached before using this function.
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RxNodeDefinitionGetPS2AllMat
 * \see RxNodeDefinitionGetPS2All
 * \see RxNodeDefinitionGetPS2Manager
 * \see RpMaterialGetDefaultRenderPipeline
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 */
RxPipeline *
RpMaterialSkySetDefaultPS2AllMatPipeline(RxPipeline *pipeline)
{
    RWAPIFUNCTION(RWSTRING("RpMaterialSkySetDefaultPS2AllMatPipeline"));

    RWASSERT(NULL != pipeline);
    if (NULL != pipeline)
    {
        RxPipelineNode *node;
        /* PS2All material pipes are NOT real, the pipe should
         * contain one and only one node, the PS2AllMat node.
         * PVS can ####ing well modify the object pipe. */
        RWASSERT(1 == pipeline->numNodes);
        node = &(pipeline->nodes[0]);
        RWASSERT(NULL != node);

        /* More paranoia */
        RWASSERT(NULL != node->nodeDef);
        RWASSERT(PS2ALLMATPRIVATEDATASIZE ==
                 node->nodeDef->pipelineNodePrivateDataSize);
        RWASSERT(NULL != node->privateData);
        RWASSERT(PS2ALLMATMAGICVALUE ==
                 ((rxNodePS2AllMatPvtData *)(node->privateData))->magicValue);

        RXPIPELINEGLOBAL(currentPS2AllMatMaterialPipeline) = pipeline;
    }

    RWRETURN(pipeline);
}

/**
 * \ingroup rpmaterialsky2
 * \page RpMaterialGetDefaultRenderPipelineplatform RpMaterialGetDefaultRenderPipeline (platform-specific)
 *
 * The default material render pipeline for PS2 is shown below.
 * Note that this is designed specifically to use VU1 to perform
 * transformation and lighting (it passes 3D triangles, through
 * PS2MatBridge.csl, to VU1). If you wish to submit 2D triangles through
 * Submit.csl (i.e you want to do transformation and lighting and whatever
 * else CPU-side - which will of course be very slow) then you should
 * replace this default material render pipeline with something based on
 * the generic material render pipeline. You should also replace the
 * associated world-sector/atomic/im3d instancing pipeline, which by
 * default instances into a PS2-specific format appropriate for this
 * pipeline, with something based on the generic instance pipelines.
 *
 * Currently, the default atomic/world-sector instance pipelines use the
 * PS2Manager node, which renders objects using just that node (for speed)
 * and ignoring material pipelines. To have your material pipelines used,
 * either incorporate them into AllInOne-style instance pipelines (see
 * \ref RpWorldSectorSkyGetAllInOnePipeline or
 * \ref RpAtomicSkyGetAllInOnePipeline) and replace the instance pipelines
 * of your objects with them, or replace the instance pipeline with a non
 * AllInOne and non PS2Manager (see \ref RpWorldSectorSkyGetPS2ManagerPipeline
 * or \ref RpAtomicSkyGetPS2ManagerPipeline) one, like that retrieved through
 * \ref RpWorldSectorSkyGetVanillaPipeline or
 * \ref RpAtomicSkyGetVanillaPipeline. You could also use generic pipelines
 * (see \ref RpMaterialGetGenericRenderPipeline, etc) but these would be
 * very slow on PS2.
 *
 * Note that you may write your own VU1 code and attach it to a material
 * pipeline so that you can perform whatever special effects you like on
 * VU1 (see the MatBridge.csl API for details).
 *
 * The world plugin must be attached before using this function.
 *
 * The default material render pipeline:
 * \verbatim
     PS2MatInstance.csl
      v
     PS2MatBridge.csl
   \endverbatim
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RxNodeDefinitionGetPS2MatInstance
 * \see RxNodeDefinitionGetPS2MatBridge
 * \see RpMaterialGetDefaultRenderPipeline
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 */
void
_rpDestroyPlatformMaterialPipelines(void)
{
    RWFUNCTION(RWSTRING("_rpDestroyPlatformMaterialPipelines"));

    /* Restore the generic pipe as default */
    RpMaterialSetDefaultRenderPipeline((RxPipeline *)NULL);

    if (NULL != RXPIPELINEGLOBAL(platformMaterialPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(platformMaterialPipeline));
        RXPIPELINEGLOBAL(platformMaterialPipeline) = (RxPipeline *)NULL;
    }

    if (NULL != RXPIPELINEGLOBAL(ps2AllMatMaterialPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(ps2AllMatMaterialPipeline));
        RXPIPELINEGLOBAL(ps2AllMatMaterialPipeline) = (RxPipeline *)NULL;
    }
    RXPIPELINEGLOBAL(currentPS2AllMatMaterialPipeline) = (RxPipeline *)NULL;

    RWRETURNVOID();
}

RwBool
_rpCreatePlatformMaterialPipelines(void)
{
    RWFUNCTION(RWSTRING("_rpCreatePlatformMaterialPipelines"));

    /* NOTE: The default material pipeline is always the same, but if the
     *       PS2Manager or AllInOne atomic/worldsector instance pipelines
     *       are used then they will ignore material pipelines. To use
     *       overloaded material pipelines under these circumstances,
     *       you must either incorporate the material pipeline into a
     *       AllInOne-style instance pipe, or replace the instance
     *       pipe with with a non-AllInOne and non-PS2Manager one.
     *       Our final paradigm will be:
     *           all pipelines are material pipelines, which act on lists
     *           of meshes. Each mesh can have (standard/plugin) data
     *           (e.g texture(s)) to differentiate it from other meshes
     *           rendered with the same pipeline. Atomics/WorldSectors
     *           have render callbacks that group meshes together based
     *           on their material pipelines (and pass in to the pipe a
     *           pointer to the parent object). */

    if ((RXPIPELINEGLOBAL(ps2AllMatMaterialPipeline) =
             CreatePS2AllMatMaterialPipeline()) != NULL)
    {
        if ((RXPIPELINEGLOBAL(platformMaterialPipeline) =
                 CreateVanillaMaterialPipeline()) != NULL)
        {
            RpMaterialSetDefaultRenderPipeline(
                RXPIPELINEGLOBAL(platformMaterialPipeline));

            /* Set up the default PS2AllMat pipe */
            RXPIPELINEGLOBAL(currentPS2AllMatMaterialPipeline)
                = RXPIPELINEGLOBAL(ps2AllMatMaterialPipeline);

            RWRETURN(TRUE);
        }
    }

    /* This should clean up safely */
    _rpDestroyPlatformMaterialPipelines();

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "PS2 material pipes"));

    RWRETURN(FALSE);
}

/**
 * \ingroup rpworldsubsky2
 * \page RpWorldGetDefaultSectorInstancePipelineplatform RpWorldGetDefaultSectorInstancePipeline (platform-specific)
 *
 * Retrieves the default world sector instance pipeline for PS2. Currently,
 * this uses the PS2Manger node for speed - see
 * \ref RpWorldSectorSkyGetPS2ManagerPipeline and
 * \ref RpMaterialGetDefaultRenderPipeline for details.
 *
 * The world plugin must be attached before using this function.
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 * \see RpAtomicGetDefaultInstancePipeline
 * \see \ref RpAtomicGetDefaultInstancePipelineplatform
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 */
void
_rpDestroyPlatformWorldSectorPipelines(void)
{
    RWFUNCTION(RWSTRING("_rpDestroyPlatformWorldSectorPipelines"));

    /* Restore the generic pipe as default */
    RpWorldSetDefaultSectorInstancePipeline((RxPipeline *)NULL);

    if (NULL != RXPIPELINEGLOBAL(vanillaWorldSectorPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(vanillaWorldSectorPipeline));
        RXPIPELINEGLOBAL(vanillaWorldSectorPipeline)= (RxPipeline *)NULL;
    }

    if (NULL != RXPIPELINEGLOBAL(allInOneWorldSectorPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(allInOneWorldSectorPipeline));
        RXPIPELINEGLOBAL(allInOneWorldSectorPipeline)= (RxPipeline *)NULL;
    }

    if (NULL != RXPIPELINEGLOBAL(platformWorldSectorPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(platformWorldSectorPipeline));
        RXPIPELINEGLOBAL(platformWorldSectorPipeline) = (RxPipeline *)NULL;
    }

    if (NULL != RXPIPELINEGLOBAL(ps2AllWorldSectorPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(ps2AllWorldSectorPipeline));
        RXPIPELINEGLOBAL(ps2AllWorldSectorPipeline) = (RxPipeline *)NULL;
    }

    RWRETURNVOID();
}

RwBool
_rpCreatePlatformWorldSectorPipelines(void)
{
    RWFUNCTION(RWSTRING("_rpCreatePlatformWorldSectorPipelines"));

    if ((RXPIPELINEGLOBAL(ps2AllWorldSectorPipeline) =
             CreatePS2AllWorldSectorPipeline()) != NULL)
    {
        if ((RXPIPELINEGLOBAL(platformWorldSectorPipeline) =
                 CreatePS2ManagerWorldSectorPipeline()) != NULL)
        {
            if ((RXPIPELINEGLOBAL(allInOneWorldSectorPipeline) =
                     CreateAllInOneWorldSectorPipeline()) != NULL)
            {
                if ((RXPIPELINEGLOBAL(vanillaWorldSectorPipeline) =
                         CreateVanillaWorldSectorPipeline()) != NULL)
                {
                    RpWorldSetDefaultSectorInstancePipeline(
                        RXPIPELINEGLOBAL(platformWorldSectorPipeline));

                    RWRETURN(TRUE);
                }
            }
        }
    }

    /* This should clean up safely */
    _rpDestroyPlatformWorldSectorPipelines();

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "PS2 sector pipes"));

    RWRETURN(FALSE);
}

/**
 * \ingroup rpatomicsky2
 * \page RpAtomicGetDefaultInstancePipelineplatform RpAtomicGetDefaultInstancePipeline (platform-specific)
 *
 * Retrieves the default atomic instance pipeline for PS2. Currently,
 * this uses the PS2Manger node for speed - see
 * \ref RpAtomicSkyGetPS2ManagerPipeline and
 * \ref RpMaterialGetDefaultRenderPipelineplatform for details.
 *
 * The world plugin must be attached before using this function.
 *
 * \return Returns pointer to the pipeline if successful or NULL if there
 * is an error.
 *
 * \see RpAtomicGetDefaultInstancePipeline
 * \see RpAtomicSetDefaultInstancePipeline
 * \see RpAtomicSkyGetPS2AllPipeline
 * \see RpAtomicSkyGetPS2ManagerPipeline
 * \see RpAtomicSkyGetAllInOnePipeline
 * \see RpAtomicSkyGetVanillaPipeline
 * \see RpAtomicGetInstancePipeline
 * \see RpAtomicSetInstancePipeline
 * \see RpMaterialGetDefaultRenderPipeline
 * \see \ref RpMaterialGetDefaultRenderPipelineplatform
 * \see RpMaterialSetDefaultRenderPipeline
 * \see RpMaterialSkyGetPS2AllMatPipeline
 * \see RpMaterialSkyGetDefaultPS2AllMatPipeline
 * \see RpMaterialSkySetDefaultPS2AllMatPipeline
 * \see RpMaterialGetRenderPipeline
 * \see RpMaterialSetRenderPipeline
 * \see RpWorldGetDefaultSectorInstancePipeline
 * \see \ref RpWorldGetDefaultSectorInstancePipelineplatform
 * \see RpWorldSetDefaultSectorInstancePipeline
 * \see RpWorldSectorSkyGetPS2AllPipeline
 * \see RpWorldSectorSkyGetPS2ManagerPipeline
 * \see RpWorldSectorSkyGetAllInOnePipeline
 * \see RpWorldSectorSkyGetVanillaPipeline
 * \see RpWorldGetSectorInstancePipeline
 * \see RpWorldSetSectorInstancePipeline
 * \see RpWorldSectorGetInstancePipeline
 * \see RpWorldSectorSetInstancePipeline
 */
void
_rpDestroyPlatformAtomicPipelines(void)
{
    RWFUNCTION(RWSTRING("_rpDestroyPlatformAtomicPipelines"));

    /* Restore the generic pipe as default */
    RpAtomicSetDefaultInstancePipeline((RxPipeline *)NULL);

    if (NULL != RXPIPELINEGLOBAL(vanillaAtomicPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(vanillaAtomicPipeline));
        RXPIPELINEGLOBAL(vanillaAtomicPipeline)= (RxPipeline *)NULL;
    }

    if (NULL != RXPIPELINEGLOBAL(allInOneAtomicPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(allInOneAtomicPipeline));
        RXPIPELINEGLOBAL(allInOneAtomicPipeline)= (RxPipeline *)NULL;
    }

    if (NULL != RXPIPELINEGLOBAL(platformAtomicPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(platformAtomicPipeline));
        RXPIPELINEGLOBAL(platformAtomicPipeline) = (RxPipeline *)NULL;
    }

    if (NULL != RXPIPELINEGLOBAL(ps2AllAtomicPipeline))
    {
        RxPipelineDestroy(RXPIPELINEGLOBAL(ps2AllAtomicPipeline));
        RXPIPELINEGLOBAL(ps2AllAtomicPipeline) = (RxPipeline *)NULL;
    }

    RWRETURNVOID();
}

RwBool
_rpCreatePlatformAtomicPipelines(void)
{
    RWFUNCTION(RWSTRING("_rpCreatePlatformAtomicPipelines"));

    if ((RXPIPELINEGLOBAL(ps2AllAtomicPipeline) =
             CreatePS2AllAtomicPipeline()) != NULL)
    {
        if ((RXPIPELINEGLOBAL(platformAtomicPipeline) =
                 CreatePS2ManagerAtomicPipeline()) != NULL)
        {
            if ((RXPIPELINEGLOBAL(allInOneAtomicPipeline) =
                     CreateAllInOneAtomicPipeline()) != NULL)
            {
                if ((RXPIPELINEGLOBAL(vanillaAtomicPipeline) =
                         CreateVanillaAtomicPipeline()) != NULL)
                {
                    RpAtomicSetDefaultInstancePipeline(
                        RXPIPELINEGLOBAL(platformAtomicPipeline));

                    RWRETURN(TRUE);
                }
            }
        }
    }

    /* This should safely clean up */
    _rpDestroyPlatformAtomicPipelines();

    RWERROR((E_RW_DEFAULTPIPELINECREATION, "PS2 stomic pipes"));

    RWRETURN(FALSE);
}

