/*
 * Memory tracking plugin
 */

/**********************************************************************
 *
 * File :     rpmeminf.h
 *
 * Abstract : Memory tracking support
 *
 **********************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd. or
 * Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. will not, under any
 * circumstances, be liable for any lost revenue or other damages arising
 * from the use of this file.
 *
 * Copyright (c) 1998 Criterion Software Ltd.
 * All Rights Reserved.
 *
 * RenderWare is a trademark of Canon Inc.
 *
 ************************************************************************/

#ifndef RPMEMINFO_H
#define RPMEMINFO_H

/**
 * \defgroup rpmeminfo RpMemInfo
 * \ingroup retired
 *
 * Memory Tracking Plug-In for RenderWare.
 */

/*--- Include files ---*/
#include <rwcore.h>
#include <rpworld.h>

#include "rpmeminf.rpe"        /* automatically generated header file */

/*--- Global Structures ---*/

typedef struct RpMemInfoBlockDescriptor RpMemInfoBlockDescriptor;
/**
 * \ingroup rpmeminfo
 * \struct RpMemInfoBlockDescriptor 
 *  structure describing a  a memory allocation 
 */
struct RpMemInfoBlockDescriptor
{
    void           *addr; /**< pointer to allocated memory */
    size_t          size; /**< size of allocated memory */
    RwUInt32        allocatorID; /**< allocator identifier */
};

typedef struct RpMemInfoAllocatorStack RpMemInfoAllocatorStack;
/**
 * \ingroup rpmeminfo
 * \struct RpMemInfoAllocatorStack 
 *  structure describing a  a memory allocation stack 
 */
struct RpMemInfoAllocatorStack
{
    RwUInt32       *stack;      /**< base of stack */
    RwUInt32        stackSize;  /**< size of stack */
    RwUInt32        top;        /**< top of stack */
};

typedef struct RpMemInfoList RpMemInfoList;
/**
 * \ingroup rpmeminfo
 * \struct RpMemInfoList 
 *  structure listing memory allocations 
 */
struct RpMemInfoList
{
    RpMemInfoBlockDescriptor *memBlocks; /**< array of memory allocations */
    RwUInt32        numBlocks;      /**< occupied length of array */
    RwUInt32        numAllocatedBlocks; /**< allocated length for array */
    RpMemInfoAllocatorStack allocatorStack; /**< allocation stack */
    RwUInt32        currentUsage;    /**< current usage */
    RwUInt32        maxUsage;        /**< hight water mark usage */
};

/**
 * \ingroup rpmeminfo
 * \typedef RpMemInfoCallBack 
 *  typedef for a callback e.g for 
 * reporting memory allocations */
typedef void   *(*RpMemInfoCallBack) (void *addr, size_t size);

/**
 * \ingroup rpmeminfo
 * \typedef RpMemInfoStatusCallBack 
 *  typedef for a callback e.g for 
 * reporting memory allocations togther with allocator identifier */
typedef void   *(*RpMemInfoStatusCallBack) (void *addr, size_t size,
                                            RwUInt32 allocatorID);

/*--- Constants ---*/

/*--- Plugin API Functions ---*/

#ifdef    __cplusplus
extern          "C"
{
#endif         /* __cplusplus */

extern RwBool          
RpMemInfoPluginAttach(void);

extern RpMemInfoList  *
RpMemInfoStartTrace(RpMemInfoList * list);

extern RpMemInfoList  *
RpMemInfoEndTrace(RpMemInfoList * list,
                  RpMemInfoCallBack callback);

extern RpMemInfoList  *
RpMemInfoPushAllocatorID(RpMemInfoList * list,
                         RwUInt32 allocatorID);

extern RwUInt32        
RpMemInfoPopAllocatorID(RpMemInfoList * list);

extern RpMemInfoList  *
RpMemInfoGetMemoryStatus(RpMemInfoList * list,
                         RpMemInfoStatusCallBack callback);

extern RwInt32         
RpMemInfoGetCurrentMemoryUsage(RpMemInfoList * list);

extern RwInt32         
RpMemInfoGetMaxMemoryUsage(RpMemInfoList * list);

extern RwInt32         
RpMemInfoGetAllocatorMemoryUsage(RpMemInfoList * list,
                                 RwUInt32 allocatorID);

extern RwInt32        
RpMemInfoGetAllocatorGroupMemoryUsage(RpMemInfoList * list,
                                      RwUInt32 allocatorID);

extern RwInt32         
RpMemInfoGetAllocatorNumBlocks(RpMemInfoList * list,
                               RwUInt32 allocatorID);

extern RwInt32         
RpMemInfoGetAllocatorGroupNumBlocks(RpMemInfoList * list,
                                    RwUInt32 allocatorID);

#ifdef    __cplusplus
}
#endif         /* __cplusplus */

#endif  /* RPMEMINFO_H */

