/*
 * Data structures for rtbezpat toolkit
 * Copyright (c) Criterion Software Limited
 */

#if (!defined(RTBEZPAT_H))
#define RTBEZPAT_H

/**
 * \defgroup rtbezpatch RtBezier
 * \ingroup rttool
 *
 * Bezier Patches
 */

/**
 * \ingroup rtbezpatch
 * \typedef RtBezierRow
 *  typedef for a difference row.
 * RtBezierRow is an  array of 4 difference vectors
 */
typedef RwV4d       RtBezierRow[4];

/**
 * \ingroup rtbezpatch
 * \typedef RtBezierMatrix
 *  typedef for a difference matrix
 *  RtBezierMatrix ia an array of 4 difference rows
 */
typedef RtBezierRow RtBezierMatrix[4];

#define RtBezierQuadMatrixSampleMacro(_out, _P, _u, _v)                 \
MACRO_START                                                             \
{                                                                       \
    const RwV4d       * const P0 = &(_P)[0][0];                         \
    const RwV4d       * const P1 = &(_P)[1][0];                         \
    const RwV4d       * const P2 = &(_P)[2][0];                         \
    const RwV4d       * const P3 = &(_P)[3][0];                         \
                                                                        \
    RtBezierRow         A;                                              \
                                                                        \
    A[0].x =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*(3*P1[3].x - P0[3].x - 3*P2[3].x + P3[3].x) +           \
           (3*P0[3].x - 6*P1[3].x + 3*P2[3].x)) +                       \
          (3*P1[3].x - 3*P0[3].x)) + P0[3].x);                          \
    A[0].y =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[3].y - P0[3].y - 3*P2[3].y +                           \
            P3[3].y) + (3*P0[3].y - 6*P1[3].y + 3*P2[3].y)) +           \
          (3*P1[3].y - 3*P0[3].y)) + P0[3].y);                          \
    A[0].z =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[3].z - P0[3].z - 3*P2[3].z +                           \
            P3[3].z) + (3*P0[3].z - 6*P1[3].z + 3*P2[3].z)) +           \
          (3*P1[3].z - 3*P0[3].z)) + P0[3].z);                          \
    A[1].x =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[2].x - P0[2].x - 3*P2[2].x + P3[2].x) +                \
           (3*P0[2].x - 6*P1[2].x + 3*P2[2].x)) +                       \
          (3*P1[2].x - 3*P0[2].x)) + P0[2].x);                          \
    A[1].y =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[2].y - P0[2].y - 3*P2[2].y + P3[2].y) +                \
           (3*P0[2].y - 6*P1[2].y + 3*P2[2].y)) +                       \
          (3*P1[2].y - 3*P0[2].y)) + P0[2].y);                          \
    A[1].z =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[2].z - P0[2].z - 3*P2[2].z + P3[2].z) +                \
           (3*P0[2].z - 6*P1[2].z + 3*P2[2].z)) +                       \
          (3*P1[2].z - 3*P0[2].z)) + P0[2].z);                          \
    A[2].x =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[1].x - P0[1].x - 3*P2[1].x + P3[1].x) +                \
           (3*P0[1].x - 6*P1[1].x + 3*P2[1].x)) +                       \
          (3*P1[1].x - 3*P0[1].x)) + P0[1].x);                          \
    A[2].y =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[1].y - P0[1].y - 3*P2[1].y + P3[1].y) +                \
           (3*P0[1].y - 6*P1[1].y + 3*P2[1].y)) +                       \
          (3*P1[1].y - 3*P0[1].y)) + P0[1].y);                          \
    A[2].z =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[1].z - P0[1].z - 3*P2[1].z + P3[1].z) +                \
           (3*P0[1].z - 6*P1[1].z + 3*P2[1].z)) +                       \
          (3*P1[1].z - 3*P0[1].z)) + P0[1].z);                          \
    A[3].x =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[0].x - P0[0].x - 3*P2[0].x + P3[0].x) +                \
           (3*P0[0].x - 6*P1[0].x + 3*P2[0].x)) +                       \
          (3*P1[0].x - 3*P0[0].x)) + P0[0].x);                          \
    A[3].y =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[0].y - P0[0].y - 3*P2[0].y + P3[0].y) +                \
           (3*P0[0].y - 6*P1[0].y + 3*P2[0].y)) +                       \
          (3*P1[0].y - 3*P0[0].y)) + P0[0].y);                          \
    A[3].z =                                                            \
        ((_u)*                                                          \
         ((_u)*                                                         \
          ((_u)*                                                        \
           (3*P1[0].z - P0[0].z - 3*P2[0].z + P3[0].z) +                \
           (3*P0[0].z - 6*P1[0].z + 3*P2[0].z)) +                       \
          (3*P1[0].z - 3*P0[0].z)) + P0[0].z);                          \
                                                                        \
    (_out)->x =                                                         \
        ((_v)*                                                          \
         ((_v)*                                                         \
          ((_v)*(3*A[2].x - A[3].x - 3*A[1].x + A[0].x) +               \
           (3*A[3].x - 6*A[2].x + 3*A[1].x)) +                          \
          (3*A[2].x - 3*A[3].x)) + A[3].x);                             \
    (_out)->y =                                                         \
        ((_v)*                                                          \
         ((_v)*                                                         \
          ((_v)*(3*A[2].y - A[3].y - 3*A[1].y + A[0].y) +               \
           (3*A[3].y - 6*A[2].y + 3*A[1].y)) +                          \
          (3*A[2].y - 3*A[3].y)) + A[3].y);                             \
    (_out)->z =                                                         \
        ((_v)*                                                          \
         ((_v)*                                                         \
          ((_v)*(3*A[2].z - A[3].z - 3*A[1].z + A[0].z) +               \
           (3*A[3].z - 6*A[2].z + 3*A[1].z)) +                          \
          (3*A[2].z - 3*A[3].z)) + A[3].z);                             \
}                                                                       \
MACRO_STOP

#define RtBezierQuadMatrixEvaluateMacro(_out, _B, _u, _v)               \
MACRO_START                                                             \
{                                                                       \
    const RwV4d       * const B0 = &(_B)[0][0];                         \
    const RwV4d       * const B1 = &(_B)[1][0];                         \
    const RwV4d       * const B2 = &(_B)[2][0];                         \
    const RwV4d       * const B3 = &(_B)[3][0];                         \
                                                                        \
    (_out)->x =                                                         \
        (((((((B0[0].x)*(_v) + B0[1].x)*(_v) +                          \
             B0[2].x)*(_v) + B0[3].x))*(_u) +                           \
          ((((B1[0].x)*(_v) + B1[1].x)*(_v) +                           \
            B1[2].x)*(_v) + B1[3].x))*(_u) +                            \
         ((((B2[0].x)*(_v) + B2[1].x)*(_v) +                            \
           B2[2].x)*(_v) + B2[3].x))*(_u) +                             \
        ((((B3[0].x)*(_v) +                                             \
           B3[1].x)*(_v) +                                              \
          B3[2].x)*(_v) +                                               \
         B3[3].x);                                                      \
                                                                        \
    (_out)->y =                                                         \
        (((((((B0[0].y)*(_v) + B0[1].y)*(_v) +                          \
             B0[2].y)*(_v) + B0[3].y))*(_u) +                           \
          ((((B1[0].y)*(_v) + B1[1].y)*(_v) +                           \
            B1[2].y)*(_v) + B1[3].y))*(_u) +                            \
         ((((B2[0].y)*(_v) + B2[1].y)*(_v) +                            \
           B2[2].y)*(_v) + B2[3].y))*(_u) +                             \
        ((((B3[0].y)*(_v) +                                             \
           B3[1].y)*(_v) +                                              \
          B3[2].y)*(_v) +                                               \
         B3[3].y);                                                      \
                                                                        \
    (_out)->z =                                                         \
        (((((((B0[0].z)*(_v) + B0[1].z)*(_v) +                          \
             B0[2].z)*(_v) + B0[3].z))*(_u) +                           \
          ((((B1[0].z)*(_v) + B1[1].z)*(_v) +                           \
            B1[2].z)*(_v) + B1[3].z))*(_u) +                            \
         ((((B2[0].z)*(_v) + B2[1].z)*(_v) +                            \
           B2[2].z)*(_v) + B2[3].z))*(_u) +                             \
        ((((B3[0].z)*(_v) +                                             \
           B3[1].z)*(_v) +                                              \
          B3[2].z)*(_v) +                                               \
         B3[3].z);                                                      \
                                                                        \
}                                                                       \
MACRO_STOP

#define RtBezierQuadMatrixEvaluateV4dMacro(_out, _B, _u, _v)            \
MACRO_START                                                             \
{                                                                       \
    const RwV4d      * const B0 = &(_B)[0][0];                          \
    const RwV4d      * const B1 = &(_B)[1][0];                          \
    const RwV4d      * const B2 = &(_B)[2][0];                          \
    const RwV4d      * const B3 = &(_B)[3][0];                          \
    RtBezierQuadMatrixEvaluateMacro((_out), (_B), (_u), (_v));          \
                                                                        \
    (_out)->w =                                                         \
        (((((((B0[0].w)*(_v) + B0[1].w)*(_v) +                          \
             B0[2].w)*(_v) + B0[3].w))*(_u) +                           \
          ((((B1[0].w)*(_v) + B1[1].w)*(_v) +                           \
            B1[2].w)*(_v) + B1[3].w))*(_u) +                            \
         ((((B2[0].w)*(_v) + B2[1].w)*(_v) +                            \
           B2[2].w)*(_v) + B2[3].w))*(_u) +                             \
        ((((B3[0].w)*(_v) +                                             \
           B3[1].w)*(_v) +                                              \
          B3[2].w)*(_v) +                                               \
         B3[3].w);                                                      \
}                                                                       \
MACRO_STOP

#define RtBezierQuadDifferenceStepUMacro(_row)                          \
MACRO_START                                                             \
{                                                                       \
    (_row)[0].x += (_row)[1].x;                                         \
    (_row)[0].y += (_row)[1].y;                                         \
    (_row)[0].z += (_row)[1].z;                                         \
    (_row)[0].w += (_row)[1].w;                                         \
                                                                        \
    (_row)[1].x += (_row)[2].x;                                         \
    (_row)[1].y += (_row)[2].y;                                         \
    (_row)[1].z += (_row)[2].z;                                         \
    (_row)[1].w += (_row)[2].w;                                         \
                                                                        \
    (_row)[2].x += (_row)[3].x;                                         \
    (_row)[2].y += (_row)[3].y;                                         \
    (_row)[2].z += (_row)[3].z;                                         \
    (_row)[2].w += (_row)[3].w;                                         \
                                                                        \
}                                                                       \
MACRO_STOP

#define RtBezierQuadDifferenceStepVMacro(_mat)                          \
MACRO_START                                                             \
{                                                                       \
    RwV4d      * const m0 = &(_mat)[0][0];                              \
    RwV4d      * const m1 = &(_mat)[1][0];                              \
    RwV4d      * const m2 = &(_mat)[2][0];                              \
    RwV4d      * const m3 = &(_mat)[3][0];                              \
                                                                        \
    /* (_row) 0 */                                                      \
    m0[0].x += m1[0].x;                                                 \
    m0[0].y += m1[0].y;                                                 \
    m0[0].z += m1[0].z;                                                 \
    m0[0].w += m1[0].w;                                                 \
                                                                        \
    m0[1].x += m1[1].x;                                                 \
    m0[1].y += m1[1].y;                                                 \
    m0[1].z += m1[1].z;                                                 \
    m0[1].w += m1[1].w;                                                 \
                                                                        \
    m0[2].x += m1[2].x;                                                 \
    m0[2].y += m1[2].y;                                                 \
    m0[2].z += m1[2].z;                                                 \
    m0[2].w += m1[2].w;                                                 \
                                                                        \
    m0[3].x += m1[3].x;                                                 \
    m0[3].y += m1[3].y;                                                 \
    m0[3].z += m1[3].z;                                                 \
    m0[3].w += m1[3].w;                                                 \
                                                                        \
    /* (_row) 1 */                                                      \
    m1[0].x += m2[0].x;                                                 \
    m1[0].y += m2[0].y;                                                 \
    m1[0].z += m2[0].z;                                                 \
    m1[0].w += m2[0].w;                                                 \
                                                                        \
    m1[1].x += m2[1].x;                                                 \
    m1[1].y += m2[1].y;                                                 \
    m1[1].z += m2[1].z;                                                 \
    m1[1].w += m2[1].w;                                                 \
                                                                        \
    m1[2].x += m2[2].x;                                                 \
    m1[2].y += m2[2].y;                                                 \
    m1[2].z += m2[2].z;                                                 \
    m1[2].w += m2[2].w;                                                 \
                                                                        \
    m1[3].x += m2[3].x;                                                 \
    m1[3].y += m2[3].y;                                                 \
    m1[3].z += m2[3].z;                                                 \
    m1[3].w += m2[3].w;                                                 \
                                                                        \
    /* (_row) 2 */                                                      \
    m2[0].x += m3[0].x;                                                 \
    m2[0].y += m3[0].y;                                                 \
    m2[0].z += m3[0].z;                                                 \
    m2[0].w += m3[0].w;                                                 \
                                                                        \
    m2[1].x += m3[1].x;                                                 \
    m2[1].y += m3[1].y;                                                 \
    m2[1].z += m3[1].z;                                                 \
    m2[1].w += m3[1].w;                                                 \
                                                                        \
    m2[2].x += m3[2].x;                                                 \
    m2[2].y += m3[2].y;                                                 \
    m2[2].z += m3[2].z;                                                 \
    m2[2].w += m3[2].w;                                                 \
                                                                        \
    m2[3].x += m3[3].x;                                                 \
    m2[3].y += m3[3].y;                                                 \
    m2[3].z += m3[3].z;                                                 \
    m2[3].w += m3[3].w;                                                 \
}                                                                       \
MACRO_STOP

#define RtBezierTriMatrixEvaluateMacro(_out, _P, _u, _v)                \
MACRO_START                                                             \
{                                                                       \
    const RwV4d      * const P0 = &(_P)[0][0];                          \
    const RwV4d      * const P1 = &(_P)[1][0];                          \
    const RwV4d      * const P2 = &(_P)[2][0];                          \
    const RwV4d      * const P3 = &(_P)[3][0];                          \
                                                                        \
    const RwReal        u1 = (_u);                                      \
    const RwReal        u2 = u1*u1;                                     \
    const RwReal        u3 = u2*u1;                                     \
    const RwReal        v1 = (_v);                                      \
    const RwReal        v2 = v1*v1;                                     \
    const RwReal        v3 = v2*v1;                                     \
                                                                        \
    (_out)->x =                                                         \
        (3*P1[0].x - P0[0].x - 3*P2[0].x + P3[0].x)*u3                  \
        + (v1*(3*P0[1].x - 3*P0[0].x + 6*P1[0].x                        \
               - 6*P1[1].x - 3*P2[0].x + 3*P2[1].x) +                   \
           3*P0[0].x - 6*P1[0].x + 3*P2[0].x)*u2                        \
        + u1*((6*P0[1].x - 3*P0[0].x - 3*P0[2].x                        \
               + 3*P1[0].x - 6*P1[1].x + 3*P1[2].x)*v2 +                \
              v1*(6*P0[0].x - 6*P0[1].x - 6*P1[0].x + 6*P1[1].x) -      \
              3*P0[0].x + 3*P1[0].x)                                    \
        + (3*P0[1].x - P0[0].x - 3*P0[2].x + P0[3].x)*v3 +              \
        (3*P0[0].x - 6*P0[1].x + 3*P0[2].x)*v2 +                        \
        v1*(3*P0[1].x - 3*P0[0].x) + P0[0].x;                           \
                                                                        \
    (_out)->y =                                                         \
        (3*P1[0].y - P0[0].y - 3*P2[0].y + P3[0].y)*u3                  \
        + (v1*(3*P0[1].y - 3*P0[0].y + 6*P1[0].y                        \
               - 6*P1[1].y - 3*P2[0].y + 3*P2[1].y) +                   \
           3*P0[0].y - 6*P1[0].y + 3*P2[0].y)*u2                        \
        + u1*((6*P0[1].y - 3*P0[0].y - 3*P0[2].y                        \
               + 3*P1[0].y - 6*P1[1].y + 3*P1[2].y)*v2 +                \
              v1*(6*P0[0].y - 6*P0[1].y - 6*P1[0].y + 6*P1[1].y) -      \
              3*P0[0].y + 3*P1[0].y)                                    \
        + (3*P0[1].y - P0[0].y - 3*P0[2].y + P0[3].y)*v3 +              \
        (3*P0[0].y - 6*P0[1].y + 3*P0[2].y)*v2 +                        \
        v1*(3*P0[1].y - 3*P0[0].y) + P0[0].y;                           \
                                                                        \
    (_out)->z =                                                         \
        (3*P1[0].z - P0[0].z - 3*P2[0].z + P3[0].z)*u3                  \
        + (v1*(3*P0[1].z - 3*P0[0].z + 6*P1[0].z                        \
               - 6*P1[1].z - 3*P2[0].z + 3*P2[1].z) +                   \
           3*P0[0].z - 6*P1[0].z + 3*P2[0].z)*u2                        \
        + u1*((6*P0[1].z - 3*P0[0].z - 3*P0[2].z                        \
               + 3*P1[0].z - 6*P1[1].z + 3*P1[2].z)*v2 +                \
              v1*(6*P0[0].z - 6*P0[1].z - 6*P1[0].z + 6*P1[1].z) -      \
              3*P0[0].z + 3*P1[0].z)                                    \
        + (3*P0[1].z - P0[0].z - 3*P0[2].z + P0[3].z)*v3 +              \
        (3*P0[0].z - 6*P0[1].z + 3*P0[2].z)*v2 +                        \
        v1*(3*P0[1].z - 3*P0[0].z) + P0[0].z;                           \
}                                                                       \
MACRO_STOP

#define RtBezierTriMatrixEvaluateV4dMacro(_out, _P, _u, _v)             \
MACRO_START                                                             \
{                                                                       \
    const RwV4d      * const P0 = &(_P)[0][0];                          \
    const RwV4d      * const P1 = &(_P)[1][0];                          \
    const RwV4d      * const P2 = &(_P)[2][0];                          \
    const RwV4d      * const P3 = &(_P)[3][0];                          \
                                                                        \
    const RwReal        u1 = (_u);                                      \
    const RwReal        u2 = u1*u1;                                     \
    const RwReal        u3 = u2*u1;                                     \
    const RwReal        v1 = (_v);                                      \
    const RwReal        v2 = v1*v1;                                     \
    const RwReal        v3 = v2*v1;                                     \
                                                                        \
    RtBezierTriMatrixEvaluateMacro((_out), (_P), (_u), (_v));           \
                                                                        \
    (_out)->w =                                                         \
        (3*P1[0].w - P0[0].w - 3*P2[0].w + P3[0].w)*u3                  \
        + (v1*(3*P0[1].w - 3*P0[0].w + 6*P1[0].w                        \
               - 6*P1[1].w - 3*P2[0].w + 3*P2[1].w) +                   \
           3*P0[0].w - 6*P1[0].w + 3*P2[0].w)*u2                        \
        + u1*((6*P0[1].w - 3*P0[0].w - 3*P0[2].w                        \
               + 3*P1[0].w - 6*P1[1].w + 3*P1[2].w)*v2 +                \
              v1*(6*P0[0].w - 6*P0[1].w - 6*P1[0].w + 6*P1[1].w) -      \
              3*P0[0].w + 3*P1[0].w)                                    \
        + (3*P0[1].w - P0[0].w - 3*P0[2].w + P0[3].w)*v3 +              \
        (3*P0[0].w - 6*P0[1].w + 3*P0[2].w)*v2 +                        \
        v1*(3*P0[1].w - 3*P0[0].w) + P0[0].w;                           \
}                                                                       \
MACRO_STOP

#define RtBezierTriDifferenceStepUMacro(_row)                           \
MACRO_START                                                             \
{                                                                       \
    (_row)[0].x += (_row)[1].x;                                         \
    (_row)[0].y += (_row)[1].y;                                         \
    (_row)[0].z += (_row)[1].z;                                         \
    (_row)[0].w += (_row)[1].w;                                         \
                                                                        \
    (_row)[1].x += (_row)[2].x;                                         \
    (_row)[1].y += (_row)[2].y;                                         \
    (_row)[1].z += (_row)[2].z;                                         \
    (_row)[1].w += (_row)[2].w;                                         \
                                                                        \
    (_row)[2].x += (_row)[3].x;                                         \
    (_row)[2].y += (_row)[3].y;                                         \
    (_row)[2].z += (_row)[3].z;                                         \
    (_row)[2].w += (_row)[3].w;                                         \
}                                                                       \
MACRO_STOP

#define RtBezierTriDifferenceStepVMacro(_mat)                           \
MACRO_START                                                             \
{                                                                       \
    RwV4d      * const m0 = &(_mat)[0][0];                              \
    RwV4d      * const m1 = &(_mat)[1][0];                              \
    RwV4d      * const m2 = &(_mat)[2][0];                              \
    RwV4d      * const m3 = &(_mat)[3][0];                              \
                                                                        \
    /* (_row) 0 */                                                      \
    m0[0].x += m1[0].x;                                                 \
    m0[0].y += m1[0].y;                                                 \
    m0[0].z += m1[0].z;                                                 \
    m0[0].w += m1[0].w;                                                 \
                                                                        \
    m0[1].x += m1[1].x;                                                 \
    m0[1].y += m1[1].y;                                                 \
    m0[1].z += m1[1].z;                                                 \
    m0[1].w += m1[1].w;                                                 \
                                                                        \
    m0[2].x += m1[2].x;                                                 \
    m0[2].y += m1[2].y;                                                 \
    m0[2].z += m1[2].z;                                                 \
    m0[2].w += m1[2].w;                                                 \
                                                                        \
                                                                        \
    m0[3].x += m1[3].x;                                                 \
    m0[3].y += m1[3].y;                                                 \
    m0[3].z += m1[3].z;                                                 \
    m0[3].w += m1[3].w;                                                 \
                                                                        \
    /* (_row) 1 */                                                      \
    m1[0].x += m2[0].x;                                                 \
    m1[0].y += m2[0].y;                                                 \
    m1[0].z += m2[0].z;                                                 \
    m1[0].w += m2[0].w;                                                 \
                                                                        \
    m1[1].x += m2[1].x;                                                 \
    m1[1].y += m2[1].y;                                                 \
    m1[1].z += m2[1].z;                                                 \
    m1[1].w += m2[1].w;                                                 \
                                                                        \
    m1[2].x += m2[2].x;                                                 \
    m1[2].y += m2[2].y;                                                 \
    m1[2].z += m2[2].z;                                                 \
    m1[2].w += m2[2].w;                                                 \
                                                                        \
    m1[3].x += m2[3].x;                                                 \
    m1[3].y += m2[3].y;                                                 \
    m1[3].z += m2[3].z;                                                 \
    m1[3].w += m2[3].w;                                                 \
                                                                        \
    /* (_row) 2 */                                                      \
    m2[0].x += m3[0].x;                                                 \
    m2[0].y += m3[0].y;                                                 \
    m2[0].z += m3[0].z;                                                 \
    m2[0].w += m3[0].w;                                                 \
                                                                        \
    m2[1].x += m3[1].x;                                                 \
    m2[1].y += m3[1].y;                                                 \
    m2[1].z += m3[1].z;                                                 \
    m2[1].w += m3[1].w;                                                 \
                                                                        \
    m2[2].x += m3[2].x;                                                 \
    m2[2].y += m3[2].y;                                                 \
    m2[2].z += m3[2].z;                                                 \
    m2[2].w += m3[2].w;                                                 \
                                                                        \
    m2[3].x += m3[3].x;                                                 \
    m2[3].y += m3[3].y;                                                 \
    m2[3].z += m3[3].z;                                                 \
    m2[3].w += m3[3].w;                                                 \
}                                                                       \
MACRO_STOP

#ifdef    __cplusplus
extern              "C"
{
#endif                          /* __cplusplus */

extern void
RtBezierQuadMatrixInitialize(RtBezierMatrix B,
                                     RtBezierMatrix P);

extern void
RtBezierQuadMatrixWeightSetup(RtBezierMatrix W,
                              RtBezierMatrix P);

extern void
RtBezierQuadMatrixDifferenceSetup(RtBezierMatrix D,
                                  RtBezierMatrix W,
                                  RwReal U,
                                  RwReal V);

extern void
RtBezierQuadMatrixDifferenceInitialize(RtBezierMatrix D,
                                       RtBezierMatrix B,
                                       RwReal U,
                                       RwReal V);

#if ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) )

extern void
RtBezierQuadMatrixSample(RwV3d * out,
                         RtBezierMatrix P,
                         RwReal u,
                         RwReal v);

extern void
RtBezierQuadMatrixEvaluate(RwV3d * out,
                           RtBezierMatrix B,
                           RwReal u,
                           RwReal v);

extern void
RtBezierQuadMatrixEvaluateV4d(RwV4d * out,
                              RtBezierMatrix B,
                              RwReal u,
                              RwReal v);

extern void
RtBezierQuadDifferenceStepU(RtBezierRow row);

extern void
RtBezierQuadDifferenceStepV(RtBezierMatrix mat);

#else  /* ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) ) */

#define RtBezierQuadMatrixEvaluate(_out, _B, _u, _v)                    \
        RtBezierQuadMatrixEvaluateMacro(_out, _B, _u, _v)

#define RtBezierQuadMatrixEvaluateV4d(_out, _B, _u, _v)                 \
        RtBezierQuadMatrixEvaluateV4dMacro(_out, _B, _u, _v)

#define RtBezierQuadDifferenceStepU(_row)                               \
        RtBezierQuadDifferenceStepUMacro(_row)

#define RtBezierQuadDifferenceStepV(_mat)                               \
        RtBezierQuadDifferenceStepVMacro(_mat)

#endif  /* ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) ) */

/*
 *
 */

extern void
RtBezierTriMatrixDifferenceInitialize(RtBezierMatrix D,
                                      RtBezierMatrix B,
                                      RwReal U, RwReal V);

extern void
RtBezierTriMatrixEvaluate(RwV3d * out,
                          RtBezierMatrix B,
                          RwReal u,
                          RwReal v);

extern void
RtBezierTriMatrixEvaluateV4d(RwV4d * out,
                             RtBezierMatrix B,
                             RwReal u,
                             RwReal v);

extern void
RtBezierTriDifferenceStepU(RtBezierRow row);

extern void
RtBezierTriDifferenceStepV(RtBezierMatrix mat);

extern void
RtBezierQuadFromTriangle(RtBezierMatrix P,
                         RtBezierMatrix T);

extern void
RtBezierQuadTangent(RtBezierMatrix Dt,
                    RtBezierMatrix Dp,
                    RwReal theta,
                    RtBezierMatrix P);

extern void
RtBezierPatchOpen(void);

#ifdef    __cplusplus
}
#endif                          /* __cplusplus */

#endif                          /* (!defined(RTBEZPAT_H)) */
