/*===========================================================================*
 *--- Include files ---------------------------------------------------------*
 *===========================================================================*/
#include <rwcore.h>
#include <rpworld.h>

#include "rpplugin.h"
#include "rpdbgerr.h"

#include "rpskin.h"
#include "rppatch.h"

#include "patch.h"
#include "patchskin.h"

#include "patchskygeneric.h"
#include "patchskyskin.h"

#include "G2_BPatch/G2_BPatchNode.h"
#include "G2_SkinBPatch/G2_SkinBPatchNode.h"

/*===========================================================================*
 *--- Private Types ---------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Private Global Variables ----------------------------------------------*
 *===========================================================================*/
RwUInt128 _rpPatchSkyLodQW = 0;

PatchGlobals _rpPatchGlobals =
{
    0,                   /* RwInt32        engineOffset                     */
    0,                   /* RwInt32        atomicOffset                     */
    0,                   /* RwInt32        geometryOffset                   */
    { 0, 0 },            /* RwModuleInfo   module                           */
    (RwFreeList *)NULL,  /* RwFreeList    *atomicFreeList                   */
    (RwFreeList *)NULL,  /* RwFreeList    *geometryFreeList                 */
    {                    /* PatchPlatform  platform                         */
        {                /* RxPipeline    *pipelines[rpPATCHSKYPIPELINEMAX] */
            (RxPipeline *)NULL, /* rpPATCHSKYPIPELINEGENERIC                */
            (RxPipeline *)NULL, /* rpPATCHSKYPIPELINEMATFX                  */
            (RxPipeline *)NULL, /* rpPATCHSKYPIPELINEMATFXUV2               */
            (RxPipeline *)NULL, /* rpPATCHSKYPIPELINESKINNED                */
            (RxPipeline *)NULL, /* rpPATCHSKYPIPELINESKINMATFX              */
            (RxPipeline *)NULL, /* rpPATCHSKYPIPELINESKINMATFXUV2           */
        }
    }
};

/*===========================================================================*
 *--- Private Defines -------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Local Types -----------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Local Global Variables ------------------------------------------------*
 *===========================================================================*/
#if (!defined(DXOYGEN))
static const char rcsid[] __RWUNUSED__ =
    "@@@@(#)$Id: patchskinsky.c,v 1.4 2001/09/26 12:26:31 johns Exp $";
#endif /* (!defined(DXOYGEN)) */

/*===========================================================================*
 *--- Local Defines ---------------------------------------------------------*
 *===========================================================================*/
#define PatchPipeline(pipeline)                                         \
    (_rpPatchGlobals.platform.pipelines[pipeline])

/*===========================================================================*
 *--- Local functions -------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Private functions -----------------------------------------------------*
 *===========================================================================*/

/****************************************************************************
 _rpSkinPipelinesCreate

 Create the patch pipelines.

 Inputs :   pipes - RwUInt32 bit-field defining which pipes to create
                        rpPATCHPIPELINEGENERIC = 0x01,
                        rpPATCHPIPELINESKINNED = 0x02,
                        rpPATCHPIPELINEMATFX   = 0x04,
 Outputs:   RwBool - on success.
 */
RwBool
_rpPatchPipelinesCreate(RwUInt32 pipes)
{
    RWFUNCTION(RWSTRING("_rpPatchPipelinesCreate"));

    /* Should we create the generic pipeline. */
    if(pipes & rpPATCHPIPELINEGENERIC)
    {
        /* Setup the vu transforms. */
        _rpPatchSkyBPatchSetupTransforms();

        /* Create the pipeline. */
        PatchPipeline(rpPATCHSKYPIPELINEGENERIC) =
            _rpPatchGenericPipelineCreate(UV);
        RWASSERT(NULL != PatchPipeline(rpPATCHSKYPIPELINEGENERIC));
    }

    /* Should we create the skinning pipeline. */
    if(pipes & rpPATCHPIPELINESKINNED)
    {
        /* Setup the vu transforms. */
        _rpPatchSkySkinBPatchSetupTransforms();

        /* Create the pipeline. */
        PatchPipeline(rpPATCHSKYPIPELINESKINNED) =
            _rpPatchSkinPipelineCreate(UV);
        RWASSERT(NULL != PatchPipeline(rpPATCHSKYPIPELINESKINNED));
    }

    RWRETURN(TRUE);
}

/****************************************************************************
 _rpPatchPipelinesDestroy

 Destroy the patch pipelines.

 Inputs :   None
 Outputs:   RwBool - on success.
 */
RwBool
_rpPatchPipelinesDestroy(void)
{
    PatchSkyPipeline iPipeline;

    RWFUNCTION(RWSTRING("_rpPatchPipelinesDestroy"));

    for( iPipeline = rpPATCHSKYPIPELINEGENERIC;
         iPipeline < rpPATCHSKYPIPELINEMAX; 
         iPipeline = (PatchSkyPipeline)(1 + (RwUInt32)iPipeline) )
    {
        if(NULL != PatchPipeline(iPipeline))
        {
            RxPipelineDestroy(PatchPipeline(iPipeline));
            PatchPipeline(iPipeline) = (RxPipeline *)NULL;
        }
        RWASSERT(NULL == PatchPipeline(iPipeline));
    }

    RWRETURN(TRUE);
}

/****************************************************************************
 _rpPatchPipelinesAttach

 Attach the correct patch pipeline to the atomic.

 Inputs :   atomic *   - Pointer to the atomic.
 Outputs:   RpAtomic * - Pointer to the atomic on success.
 */
RpAtomic *
_rpPatchPipelinesAttach(RpAtomic *atomic)
{
    RpPatchMesh *patchMesh;
    RpSkin *skin;

    PatchSkyPipeline iPipeline;
    RxPipeline *pipeline;

    RWFUNCTION(RWSTRING("_rpPatchPipelinesAttach"));
    RWASSERT(NULL != atomic);

    /* Get the patch mesh. */
    patchMesh = RpPatchAtomicGetPatchMesh(atomic);
    RWASSERT(NULL != patchMesh);

    /* Get the skin. */
    skin = RpPatchMeshGetSkin(patchMesh);

    /* Is the patch mesh skinned. */
    if(NULL != skin)
    {
        iPipeline = rpPATCHSKYPIPELINESKINNED;
    }
    else
    {
        iPipeline = rpPATCHSKYPIPELINEGENERIC;
    }

    pipeline = PatchPipeline(iPipeline);
    RWASSERT(NULL != pipeline);

    atomic = RpAtomicSetPipeline(atomic, pipeline);
    RWASSERT(NULL != atomic);

    RWRETURN(atomic);
}

/*===========================================================================*
 *--- Plugin Engine Functions -----------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Plugin API Functions --------------------------------------------------*
 *===========================================================================*/
